/*
 * Copyright (C) 2019 Andreas Steffen
 * Copyright (C) 2015 Martin Willi
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * As I couldn't find any test vectors for common MODP diffie hellman groups
 * these have been generated.
 */

ke_test_vector_t modp768 = {
	.method = MODP_768_BIT,
	.seed = chunk_from_chars(
		0x60, 0x91, 0xff, 0xc6, 0xde, 0x28, 0xc1, 0xcc, 0xc7, 0xc6,
		0x5d, 0xa6, 0x11, 0xab, 0xfa, 0xe8, 0x6a, 0x10, 0x74, 0xb2,
		0x22, 0x43, 0xe3, 0x70, 0x6c, 0xb6, 0xde, 0x2f, 0xe2, 0x9d,
		0x11, 0x42, /* - */
		0x76, 0xe0, 0x2f, 0xc3, 0xea, 0xbe, 0x6a, 0x0f, 0xce, 0xd6,
		0xc3, 0x1e, 0x59, 0x45, 0xd1, 0x67, 0xfa, 0xd0, 0x02, 0x00,
		0xb4, 0xaf, 0x12, 0xcd, 0x6f, 0xc6, 0xd7, 0xe3, 0x81, 0x48,
		0x62, 0x78),
	.pub_i = chunk_from_chars(
		0xd7, 0xa2, 0x54, 0x62, 0x99, 0x01, 0xc8, 0x89, 0x53, 0x2c,
		0x3e, 0x44, 0xda, 0x3d, 0x0b, 0x7e, 0x92, 0x27, 0x37, 0x70,
		0xc6, 0x26, 0xc3, 0x91, 0x90, 0x52, 0x2d, 0xab, 0x67, 0x07,
		0xca, 0xff, 0x36, 0x59, 0x10, 0x9f, 0x2f, 0x43, 0x24, 0xa4,
		0x17, 0xeb, 0x7d, 0xc4, 0x56, 0x3a, 0x09, 0xba, 0x04, 0xcd,
		0x3c, 0x9b, 0x27, 0xd3, 0xc1, 0x22, 0x7e, 0xb1, 0x9d, 0xcb,
		0x69, 0xfe, 0xf3, 0xb6, 0xe2, 0xaa, 0x13, 0x81, 0x23, 0x24,
		0x06, 0x64, 0x2d, 0xd1, 0x50, 0x78, 0x57, 0x07, 0xf4, 0x7c,
		0x3d, 0x74, 0x8a, 0x3d, 0x6b, 0x96, 0xd0, 0x00, 0xc5, 0x2c,
		0x99, 0xd1, 0x0b, 0x65, 0xf2, 0xd1),
	.pub_r = chunk_from_chars(
		0xf0, 0xe9, 0xdc, 0x78, 0x10, 0x4c, 0x97, 0x99, 0xb6, 0x70,
		0x74, 0xb0, 0x7d, 0x8f, 0x09, 0x7a, 0xa8, 0x82, 0xbd, 0xe4,
		0x64, 0xc8, 0xeb, 0x9a, 0x0a, 0xcd, 0xef, 0x13, 0x86, 0x19,
		0x4f, 0x49, 0xc0, 0x63, 0xc6, 0x7d, 0x68, 0xf1, 0x4c, 0x5e,
		0x3b, 0x04, 0x19, 0x57, 0x67, 0x8e, 0xa7, 0xcb, 0x39, 0x7d,
		0x87, 0x07, 0x20, 0x81, 0x9e, 0xa1, 0x08, 0x5a, 0x28, 0xd8,
		0x13, 0xe3, 0x73, 0x9a, 0x64, 0x00, 0x6f, 0x24, 0x66, 0xe8,
		0x6c, 0x09, 0xe0, 0xc6, 0x9c, 0x2c, 0xa4, 0xf1, 0x0a, 0x04,
		0xc4, 0x9b, 0xb3, 0x01, 0x86, 0xbd, 0xfa, 0xb5, 0x4d, 0xf4,
		0x20, 0x83, 0x14, 0xd9, 0x3c, 0x52),
	.shared = chunk_from_chars(
		0x29, 0xcb, 0x14, 0x7d, 0x2a, 0x2b, 0x0d, 0x02, 0x59, 0x4d,
		0xb0, 0x7b, 0xf0, 0x44, 0x70, 0x5c, 0xb1, 0x44, 0x89, 0xd0,
		0xf0, 0xa9, 0x32, 0xcc, 0x87, 0xf6, 0x04, 0x05, 0x1d, 0x1c,
		0xb6, 0xe2, 0xbe, 0x90, 0x39, 0x16, 0xe2, 0x5c, 0x65, 0x39,
		0x93, 0xc8, 0x05, 0x5e, 0xd4, 0x37, 0x6c, 0xa4, 0xdb, 0xa8,
		0x40, 0x12, 0x39, 0x1a, 0x92, 0xa3, 0xf3, 0x42, 0x64, 0xaf,
		0x64, 0x1f, 0xd8, 0x18, 0xe1, 0xba, 0x4e, 0x99, 0x34, 0x30,
		0x09, 0x97, 0xd7, 0x32, 0xd7, 0x0d, 0x0d, 0x73, 0x73, 0xcd,
		0x3d, 0xaa, 0xad, 0x22, 0xe8, 0x68, 0xb7, 0xe2, 0x50, 0x02,
		0x9d, 0x30, 0x7e, 0xe5, 0x41, 0x48)
};

ke_test_vector_t modp1024 = {
	.method = MODP_1024_BIT,
	.seed = chunk_from_chars(
		0xe5, 0x3c, 0x20, 0x5d, 0xa0, 0xd8, 0xe4, 0xbf, 0xb4, 0x17,
		0x49, 0x44, 0x32, 0x0f, 0xc6, 0xe4, 0xea, 0x66, 0xfe, 0x44,
		0xe3, 0xc9, 0x31, 0xac, 0x5d, 0xa1, 0x45, 0x0a, 0xea, 0x47,
		0xeb, 0xcf, /* - */
		0x7f, 0x9a, 0xf7, 0x21, 0xeb, 0x7c, 0xd2, 0xa9, 0x00, 0xa3,
		0x6e, 0x39, 0x9e, 0xbc, 0x5c, 0x65, 0xee, 0xcc, 0xe6, 0x62,
		0x9c, 0x8e, 0x1c, 0x5a, 0x7f, 0xf3, 0x32, 0x93, 0x64, 0x5f,
		0xd3, 0xe3),
	.pub_i = chunk_from_chars(
		0x66, 0x61, 0x7c, 0x9b, 0xbe, 0x91, 0xee, 0x99, 0x00, 0xd8,
		0x06, 0x41, 0x5b, 0x73, 0x84, 0xec, 0xb0, 0xb6, 0xb8, 0x0a,
		0x39, 0xbd, 0x5b, 0x07, 0x03, 0x96, 0xee, 0x32, 0x37, 0x5a,
		0x8f, 0x68, 0x37, 0x6b, 0x35, 0x2b, 0x97, 0xba, 0xf3, 0x2f,
		0x95, 0xc4, 0xd1, 0x8c, 0x06, 0xab, 0x96, 0xbf, 0xe3, 0xf3,
		0x75, 0x2d, 0xf1, 0xe3, 0xc5, 0x57, 0x41, 0xb6, 0xf4, 0x24,
		0x41, 0x17, 0xee, 0xbd, 0xa1, 0x01, 0x59, 0x83, 0xc6, 0xba,
		0x00, 0x8f, 0xe8, 0x9f, 0xe5, 0x1c, 0xf4, 0xc9, 0x69, 0x25,
		0x92, 0xeb, 0xf7, 0x42, 0x43, 0x6c, 0x39, 0x3d, 0xf6, 0x1a,
		0x60, 0xcc, 0xc6, 0x4f, 0xd7, 0x90, 0x7a, 0x6d, 0x26, 0x26, /* 100 */
		0x7c, 0x0f, 0x15, 0x56, 0x23, 0x0c, 0x3e, 0x33, 0x2b, 0x2e,
		0x11, 0xd1, 0x18, 0xa6, 0x98, 0x25, 0x89, 0x79, 0x3a, 0x2f,
		0x6d, 0x66, 0xb8, 0x66, 0x76, 0xba, 0xe4, 0x7a),
	.pub_r = chunk_from_chars(
		0x7b, 0x90, 0x10, 0x60, 0x9e, 0xbf, 0x2c, 0x49, 0x70, 0x17,
		0x45, 0x8a, 0xfa, 0xab, 0x42, 0x02, 0x88, 0x5d, 0x25, 0xde,
		0x7b, 0x5e, 0x5c, 0xe9, 0xb0, 0x5b, 0xd4, 0x42, 0xa3, 0xe9,
		0x7b, 0x52, 0xce, 0xa7, 0x60, 0xd7, 0xdb, 0xcb, 0x21, 0xdd,
		0x71, 0xd8, 0x0c, 0xd4, 0x34, 0x7c, 0xaa, 0x9e, 0xdf, 0xbc,
		0x2d, 0xf4, 0xc1, 0xcd, 0xc1, 0x66, 0x9b, 0x8a, 0xd2, 0x44,
		0xeb, 0x34, 0x5b, 0x33, 0x1d, 0x87, 0x54, 0x92, 0x88, 0x3d,
		0xf0, 0x4a, 0x3d, 0x0b, 0x1a, 0x8b, 0x89, 0x27, 0xd5, 0x09,
		0x91, 0xfe, 0x03, 0xe7, 0x35, 0x7e, 0xb6, 0xbd, 0xfc, 0xe3,
		0xd8, 0xc6, 0x84, 0xc9, 0x86, 0x1b, 0xc5, 0xce, 0x03, 0x96, /* 100 */
		0x2f, 0xfb, 0x87, 0xbf, 0x05, 0xda, 0xbd, 0x5a, 0x37, 0x27,
		0x99, 0x4d, 0xde, 0xe5, 0xd3, 0xc6, 0xec, 0xc5, 0x89, 0x85,
		0x99, 0x91, 0xb9, 0x32, 0x55, 0x76, 0x1f, 0xd5),
	.shared = chunk_from_chars(
		0x95, 0x79, 0x0f, 0x5c, 0x46, 0xae, 0x7c, 0xa1, 0xa4, 0x71,
		0xdd, 0x78, 0x6a, 0xa8, 0xe4, 0x44, 0x07, 0x3e, 0xce, 0xc9,
		0x69, 0x5d, 0x00, 0x46, 0x20, 0xcd, 0x7c, 0x9d, 0x36, 0x09,
		0xa6, 0x97, 0x3c, 0x89, 0xa2, 0x30, 0x75, 0x09, 0x35, 0x63,
		0x8b, 0x86, 0xd1, 0xe6, 0x5b, 0x27, 0xb7, 0x84, 0x88, 0x81,
		0xf9, 0x01, 0x3a, 0xbd, 0x03, 0x62, 0x80, 0xd1, 0x86, 0x2b,
		0xb0, 0x3c, 0xa6, 0x0b, 0xa9, 0x0b, 0x70, 0xf9, 0xae, 0x7e,
		0xdf, 0x71, 0xff, 0x80, 0xf5, 0xa7, 0xee, 0xfb, 0xe2, 0x67,
		0x29, 0xb5, 0xbb, 0xb1, 0xbb, 0x1f, 0xeb, 0x0d, 0x24, 0x0e,
		0x53, 0xc9, 0xd7, 0x4b, 0x4f, 0xe9, 0xd2, 0x62, 0xb5, 0x3b, /* 100 */
		0xd5, 0xd8, 0xa7, 0x38, 0x3f, 0x90, 0xf0, 0x1e, 0x35, 0x96,
		0x47, 0xd0, 0x48, 0x02, 0xd7, 0x4a, 0x4f, 0x75, 0x3b, 0x29,
		0x4a, 0x96, 0x50, 0x3f, 0x26, 0x05, 0xd3, 0xf1)
};

ke_test_vector_t modp1536 = {
	.method = MODP_1536_BIT,
	.seed = chunk_from_chars(
		0x22, 0xd9, 0xdc, 0xc7, 0x30, 0x79, 0x93, 0x6a, 0x85, 0x8c,
		0x07, 0xaa, 0x85, 0xed, 0x07, 0xb3, 0xd1, 0xe8, 0xb6, 0x70,
		0xe7, 0xca, 0xaf, 0xa1, 0x92, 0x83, 0x76, 0x96, 0x07, 0x0f,
		0xef, 0x29, /* - */
		0x32, 0x34, 0x74, 0xde, 0x77, 0x88, 0xe0, 0x03, 0x6b, 0x30,
		0x95, 0x49, 0x56, 0x0b, 0x00, 0x0d, 0x81, 0xf8, 0x2f, 0xdb,
		0x69, 0x78, 0xf3, 0xc0, 0x3b, 0x70, 0x16, 0x80, 0xde, 0x6e,
		0x28, 0x10),
	.pub_i = chunk_from_chars(
		0x3d, 0x7d, 0x1d, 0xd1, 0xbc, 0xa7, 0x13, 0x7a, 0x64, 0x23,
		0x73, 0xd5, 0xd9, 0xb8, 0x6e, 0xf9, 0x95, 0x84, 0x9e, 0xa5,
		0x1c, 0xb6, 0xcd, 0x51, 0xa6, 0xb9, 0x3e, 0x83, 0xb3, 0x4f,
		0x96, 0x7d, 0xbf, 0x7d, 0x66, 0xbc, 0x7c, 0xe5, 0xd9, 0x58,
		0xbf, 0x01, 0x90, 0x2b, 0x60, 0xf1, 0xc3, 0x07, 0x6c, 0xfe,
		0x14, 0x7b, 0xeb, 0x45, 0xf0, 0x83, 0x97, 0xcf, 0xf4, 0xc3,
		0xa8, 0x02, 0x7c, 0xaa, 0xe1, 0x84, 0x78, 0x8e, 0xf3, 0xeb,
		0x0d, 0xd5, 0x6e, 0x14, 0xc6, 0xdd, 0x2c, 0xe9, 0x54, 0xe3,
		0xd5, 0xcc, 0x80, 0xdb, 0x84, 0xde, 0xb5, 0x34, 0xce, 0x38,
		0x24, 0x45, 0xe7, 0xa4, 0x41, 0xdb, 0x97, 0x12, 0x30, 0x02, /* 100 */
		0x0a, 0x1a, 0x45, 0xca, 0x00, 0x70, 0xd6, 0x94, 0xf0, 0x93,
		0xc7, 0x16, 0xd8, 0x07, 0x68, 0x33, 0x6e, 0x61, 0xb5, 0x6f,
		0xf7, 0x8b, 0x35, 0x09, 0x39, 0xfe, 0x4e, 0x9e, 0x03, 0x2c,
		0x85, 0xbb, 0x58, 0x81, 0xc4, 0xc8, 0xd7, 0xdb, 0xd5, 0x30,
		0xa6, 0xfc, 0x50, 0x13, 0x00, 0xf7, 0xe9, 0xe6, 0x5b, 0xff,
		0xb9, 0x83, 0x34, 0x8a, 0xd0, 0x5c, 0xc5, 0x6e, 0x2c, 0x19,
		0xf5, 0x97, 0xa9, 0x9f, 0xb9, 0x68, 0x91, 0x4b, 0xe9, 0xb5,
		0x7a, 0xcf, 0x91, 0x11, 0xe7, 0x5b, 0x57, 0x6a, 0x61, 0x37,
		0x67, 0x15, 0x76, 0x57, 0x90, 0x0d, 0xcf, 0x22, 0xf4, 0x20,
		0x0c, 0x3d),
	.pub_r = chunk_from_chars(
		0xd6, 0x32, 0x58, 0xd5, 0x54, 0x35, 0x3d, 0x6b, 0x2b, 0xcc,
		0x0b, 0x53, 0x53, 0xfa, 0x80, 0x00, 0xb3, 0xa3, 0x54, 0xa2,
		0x41, 0x1d, 0x7f, 0x17, 0xab, 0xca, 0x69, 0x1d, 0xe8, 0x22,
		0x7c, 0xd7, 0xd4, 0x3e, 0x7f, 0xef, 0x8b, 0x3e, 0xe7, 0xa0,
		0xa9, 0x1a, 0x66, 0x3d, 0x30, 0xc5, 0x4f, 0x3c, 0x5f, 0x4b,
		0x95, 0xc9, 0xfe, 0x38, 0xc6, 0xcf, 0x19, 0x39, 0xb4, 0x80,
		0x2b, 0xb6, 0xf0, 0xa9, 0x51, 0x12, 0x8f, 0xdc, 0x39, 0x1d,
		0x90, 0xfa, 0x8b, 0x40, 0x48, 0x4f, 0x45, 0xb6, 0xda, 0x02,
		0xc7, 0x6c, 0xf9, 0x1b, 0x43, 0x31, 0xc4, 0xcf, 0x78, 0x51,
		0xe5, 0x50, 0xa2, 0xd1, 0xc0, 0x25, 0x53, 0x14, 0x03, 0xe0, /* 100 */
		0x40, 0x3a, 0xf4, 0x72, 0xb3, 0x83, 0x41, 0xdc, 0x56, 0x2b,
		0xe9, 0x8d, 0x1a, 0xa2, 0x80, 0x49, 0x4d, 0x62, 0x64, 0x31,
		0x6a, 0x6f, 0x77, 0x4c, 0xe0, 0xee, 0xd4, 0x01, 0x57, 0xb6,
		0x37, 0xc5, 0x4e, 0x69, 0x50, 0x3b, 0xec, 0xb8, 0xf0, 0xd7,
		0x57, 0xb2, 0x86, 0xe4, 0xd8, 0x43, 0xc2, 0x24, 0x21, 0x9a,
		0x92, 0x3f, 0x73, 0xab, 0x57, 0x83, 0x15, 0xaf, 0x44, 0x5a,
		0xfa, 0xd9, 0x56, 0x9d, 0xf2, 0x5a, 0xcf, 0xca, 0x3a, 0x9f,
		0x7c, 0x93, 0xd3, 0x03, 0xf4, 0xef, 0x24, 0x32, 0xbf, 0x62,
		0xce, 0x52, 0x5c, 0x8a, 0x56, 0xba, 0xbd, 0x2f, 0xfb, 0x54,
		0x38, 0x32),
	.shared = chunk_from_chars(
		0x6e, 0x3d, 0xf6, 0xe2, 0x52, 0xba, 0x11, 0x53, 0xca, 0x45,
		0xe9, 0xa8, 0xbb, 0xe5, 0x48, 0x33, 0x7b, 0x69, 0x57, 0x2a,
		0xff, 0x4e, 0x61, 0xb4, 0x6e, 0xc9, 0x86, 0xb8, 0x70, 0xa8,
		0x63, 0xd3, 0x85, 0x3a, 0xb9, 0xa5, 0x4d, 0x9c, 0x63, 0x2a,
		0x09, 0x48, 0xce, 0x8c, 0x65, 0xf1, 0xa2, 0x9b, 0x06, 0x7a,
		0x14, 0x51, 0x19, 0x8d, 0xab, 0x9f, 0x24, 0x77, 0x6e, 0x86,
		0x42, 0x5a, 0x06, 0xbd, 0xaa, 0x9f, 0x26, 0xdc, 0xe9, 0xe8,
		0x9e, 0x36, 0x8d, 0x0e, 0x16, 0x70, 0x27, 0x74, 0x01, 0x5a,
		0x14, 0x30, 0xaa, 0xa1, 0xe8, 0x33, 0x22, 0x7f, 0x9d, 0xea,
		0x7a, 0x58, 0x18, 0xce, 0x47, 0x43, 0x20, 0xb3, 0x9c, 0xe8, /* 100 */
		0x67, 0xca, 0x79, 0xa4, 0x9b, 0x31, 0xe6, 0xab, 0xce, 0xa6,
		0xc8, 0xda, 0xff, 0x00, 0xb8, 0x06, 0x4e, 0x2a, 0x75, 0x73,
		0x72, 0xd4, 0x0d, 0x58, 0xa5, 0x92, 0xe7, 0xa2, 0xde, 0xb1,
		0xf9, 0xa0, 0xd9, 0xab, 0xab, 0x1f, 0xc0, 0x81, 0x2e, 0xe1,
		0xff, 0xa6, 0x2a, 0x20, 0xff, 0x68, 0xce, 0x4d, 0x02, 0xac,
		0xb8, 0x4a, 0x1a, 0x03, 0x3d, 0x03, 0xe4, 0xf0, 0x5e, 0x97,
		0xa4, 0xfa, 0xd8, 0x9e, 0xc2, 0x3a, 0xee, 0x34, 0x9e, 0x26,
		0x4e, 0xfa, 0x61, 0xae, 0x59, 0xe9, 0x38, 0x1b, 0x1e, 0x5b,
		0x7a, 0xa5, 0xd0, 0x9a, 0xb6, 0x6b, 0x74, 0x99, 0x7c, 0xba,
		0xed, 0x20)
};

ke_test_vector_t modp2048 = {
	.method = MODP_2048_BIT,
	.seed = chunk_from_chars(
		0xaf, 0x3b, 0xfd, 0x38, 0x62, 0xca, 0xa1, 0x17, 0x74, 0xce,
		0x2b, 0x74, 0x84, 0x08, 0x07, 0xc1, 0xde, 0x5c, 0xd6, 0xa7,
		0x61, 0x9b, 0xb3, 0xa0, 0xc7, 0xaf, 0x39, 0xee, 0xda, 0xa6,
		0xeb, 0x89, 0xe2, 0xe9, 0xc1, 0x44, 0xb3, 0x62, 0x5b, 0x27,
		0x31, 0x87, 0x9c, 0xb5, 0x8f, 0xa3, 0x76, 0x6d, /* - */
		0x77, 0xa4, 0x8e, 0x47, 0x72, 0xe2, 0x3e, 0x28, 0x4c, 0xe4,
		0xaf, 0x81, 0x39, 0x9d, 0xcd, 0x58, 0x9b, 0xeb, 0x7c, 0xef,
		0xbc, 0xc9, 0xd1, 0x96, 0xf0, 0x6d, 0xcf, 0xdf, 0xc3, 0xa8,
		0x8e, 0x3b, 0x4c, 0x1c, 0x82, 0xbe, 0xfe, 0xc0, 0xe6, 0x4b,
		0xa4, 0x95, 0xcc, 0xde, 0x32, 0x99, 0x36, 0xce),
	.pub_i = chunk_from_chars(
		0x1a, 0x27, 0xc5, 0xa7, 0x23, 0x7a, 0xe8, 0xfe, 0x0a, 0x71,
		0xc0, 0xaf, 0x64, 0x94, 0xfa, 0xec, 0xcd, 0xd4, 0xa5, 0x03,
		0x1d, 0x47, 0xa1, 0xa6, 0xb7, 0x16, 0x2d, 0xfc, 0xdf, 0x03,
		0xbb, 0x17, 0xb7, 0xbb, 0x08, 0x9b, 0xd6, 0x52, 0xa5, 0xf3,
		0x1b, 0x36, 0x61, 0xbb, 0x11, 0x4d, 0x7c, 0x48, 0xd8, 0x3d,
		0x24, 0x4f, 0x46, 0xdb, 0x64, 0x87, 0x0c, 0x9c, 0x83, 0x27,
		0xf1, 0xa8, 0x30, 0xab, 0xf5, 0x31, 0xe5, 0x18, 0xdd, 0x52,
		0x1a, 0x2e, 0x94, 0xe7, 0x06, 0x1e, 0x94, 0x42, 0x09, 0xba,
		0x53, 0x31, 0x01, 0x2b, 0x3d, 0xff, 0x00, 0x84, 0x9e, 0xaa,
		0xb5, 0x8c, 0x7c, 0x7a, 0xf1, 0x52, 0x65, 0x21, 0x0f, 0xbc, /* 100 */
		0xf0, 0xf8, 0x7b, 0x9d, 0xd9, 0x32, 0xf8, 0xba, 0x4e, 0x9a,
		0x9f, 0x91, 0xb6, 0x32, 0x94, 0x53, 0x0a, 0x12, 0x00, 0xb8,
		0x8b, 0x3a, 0x03, 0xe1, 0xa0, 0xc7, 0xfd, 0x34, 0xde, 0xec,
		0x6f, 0xad, 0x50, 0x1e, 0xde, 0x06, 0xce, 0x7e, 0xc9, 0xe2,
		0x91, 0xd4, 0x7b, 0x69, 0xb3, 0xa4, 0xb6, 0x1b, 0xb0, 0x87,
		0x63, 0xb6, 0xd4, 0x6b, 0xfb, 0xda, 0xb9, 0x7b, 0xde, 0xce,
		0xa0, 0xff, 0x09, 0xa8, 0x72, 0x22, 0x1f, 0xb4, 0xb6, 0xcc,
		0xca, 0x6b, 0x5c, 0x6f, 0xaa, 0xd8, 0xeb, 0x87, 0x3b, 0x9a,
		0x2f, 0x39, 0xb6, 0x3e, 0x62, 0xcf, 0xad, 0xd4, 0x76, 0x07,
		0xe6, 0x8e, 0x2c, 0x29, 0x05, 0x83, 0xf7, 0x26, 0x8f, 0xb4, /* 200 */
		0x3f, 0xc0, 0x0a, 0x7d, 0xc8, 0x81, 0x0e, 0xcd, 0xac, 0x1a,
		0x59, 0xbd, 0x1b, 0x47, 0x7a, 0xaf, 0x41, 0xcb, 0x4b, 0x24,
		0xad, 0xa3, 0xe5, 0xf7, 0xb8, 0xcb, 0x98, 0x6f, 0x1a, 0xe4,
		0x76, 0xaf, 0xc7, 0x67, 0x7a, 0xa6, 0x25, 0x70, 0xa1, 0xb1,
		0x83, 0x8d, 0xda, 0x26, 0xbe, 0x78, 0x63, 0xee, 0x2f, 0x40,
		0xb5, 0xe5, 0x38, 0xa4, 0xe1, 0x81),
	.pub_r = chunk_from_chars(
		0x66, 0x72, 0x82, 0x31, 0x8d, 0x65, 0xcb, 0x05, 0xcd, 0x32,
		0x3c, 0x25, 0x09, 0x1e, 0x66, 0x4c, 0xc7, 0x5a, 0x3a, 0x20,
		0xc0, 0x14, 0xb2, 0xf6, 0x38, 0x9d, 0x3b, 0x27, 0xf5, 0xc1,
		0xe9, 0x08, 0xe0, 0xd4, 0x0f, 0xf8, 0x55, 0x2b, 0xd5, 0x87,
		0x75, 0x25, 0x50, 0x30, 0x22, 0x85, 0xb4, 0x9f, 0x75, 0xf8,
		0xff, 0xae, 0xc9, 0x37, 0xfb, 0x2e, 0x69, 0xc8, 0x15, 0x0f,
		0x88, 0xd6, 0x50, 0x8b, 0xf0, 0xfb, 0x1f, 0x16, 0xe9, 0x67,
		0x6f, 0x5e, 0xcf, 0x71, 0x8f, 0x87, 0x4f, 0x3d, 0x8d, 0xe3,
		0xc8, 0x68, 0x87, 0x49, 0xdb, 0xcf, 0x1a, 0xff, 0xd4, 0xd6,
		0x2f, 0xf9, 0xea, 0x5d, 0x9e, 0x30, 0x35, 0xb2, 0xce, 0x4a, /* 100 */
		0x6e, 0xae, 0x28, 0x1c, 0x40, 0x8f, 0xc8, 0x08, 0x4a, 0xe2,
		0x35, 0xd9, 0x98, 0xa0, 0x37, 0x61, 0x0a, 0x8c, 0xb4, 0x8c,
		0x87, 0x8c, 0xd0, 0x63, 0x5f, 0xa1, 0x36, 0x9f, 0xc9, 0x0b,
		0x86, 0x4c, 0x53, 0x23, 0x67, 0x38, 0x45, 0x97, 0xfd, 0xee,
		0xfd, 0xb6, 0x74, 0x2b, 0x00, 0x6a, 0xde, 0xda, 0xe2, 0x01,
		0x5b, 0xc4, 0xce, 0x2f, 0x9c, 0x54, 0x51, 0xc9, 0x63, 0x6a,
		0x16, 0xed, 0x7d, 0x10, 0xb4, 0xe4, 0x0d, 0x82, 0x5b, 0x50,
		0xaa, 0x76, 0x33, 0xa0, 0xe8, 0xfb, 0x81, 0x3b, 0xeb, 0xc0,
		0x49, 0xf7, 0xff, 0x6a, 0x71, 0xec, 0xfb, 0x14, 0xe8, 0x80,
		0xf5, 0x09, 0xd3, 0x8b, 0x57, 0x1a, 0x80, 0x98, 0x08, 0xa4, /* 200 */
		0x96, 0xb4, 0x51, 0xb5, 0xb5, 0x56, 0x7a, 0x36, 0x08, 0xb0,
		0x1a, 0x22, 0xe4, 0x99, 0x83, 0x55, 0xf1, 0x81, 0xb8, 0x79,
		0xde, 0x26, 0x23, 0x04, 0xf2, 0x9d, 0xe1, 0x42, 0xff, 0x5e,
		0x52, 0xcc, 0x56, 0x4f, 0x3a, 0x2d, 0x1e, 0x84, 0xa0, 0xc3,
		0x3d, 0x8f, 0x3d, 0xa7, 0xbf, 0x64, 0x12, 0xb3, 0xf0, 0x18,
		0xe0, 0x0d, 0x90, 0xfa, 0x5b, 0x94),
	.shared = chunk_from_chars(
		0x26, 0xf9, 0xe8, 0x45, 0x7c, 0xf2, 0xd7, 0xd5, 0x6d, 0xd8,
		0x19, 0xc2, 0xa8, 0x67, 0xd7, 0xc3, 0xa9, 0x0a, 0xa2, 0xb9,
		0x44, 0xe2, 0x0b, 0x8b, 0xde, 0xd3, 0xa5, 0xbf, 0x91, 0xaa,
		0x1f, 0xf8, 0x1a, 0x7e, 0xa8, 0x9d, 0x52, 0x4c, 0x36, 0x5f,
		0x38, 0x23, 0x51, 0x08, 0x35, 0xcd, 0x86, 0x44, 0xc4, 0xd5,
		0xda, 0xcf, 0x89, 0x3e, 0x61, 0x0d, 0x2c, 0x1a, 0x14, 0x06,
		0x16, 0xd1, 0xc8, 0xc1, 0xd3, 0x5f, 0x46, 0xde, 0x2c, 0x5f,
		0xaa, 0xd9, 0xae, 0xce, 0xc6, 0xb6, 0x77, 0xac, 0xd9, 0x84,
		0x09, 0x81, 0x42, 0xa8, 0xaa, 0x0c, 0x85, 0x42, 0xe9, 0xac,
		0x98, 0xb3, 0xbc, 0x7b, 0x57, 0x7c, 0x59, 0xb8, 0xec, 0x53, /* 100 */
		0x34, 0x6e, 0x15, 0x67, 0xa9, 0x00, 0x39, 0xac, 0x92, 0xb2,
		0x24, 0x63, 0x5a, 0xb2, 0x16, 0x73, 0x1b, 0x06, 0x35, 0x39,
		0x25, 0x64, 0x2c, 0x33, 0x0a, 0x20, 0x1b, 0xa4, 0xb3, 0x12,
		0x84, 0xa9, 0x51, 0x58, 0x60, 0xf3, 0x5e, 0x93, 0x08, 0xf5,
		0x51, 0x5a, 0x77, 0x99, 0x84, 0xfb, 0xd9, 0xf5, 0xce, 0x41,
		0x77, 0xdb, 0x78, 0xd1, 0xcb, 0x03, 0x84, 0xb6, 0x3c, 0x73,
		0x9c, 0x6d, 0x74, 0x4a, 0xd7, 0xa7, 0x00, 0xb9, 0x5a, 0x53,
		0x1b, 0x29, 0xf3, 0xb7, 0x44, 0xed, 0x38, 0xe6, 0x9a, 0xee,
		0x67, 0x07, 0x2c, 0x45, 0xa9, 0x1c, 0xee, 0x6b, 0x14, 0x21,
		0x5e, 0x04, 0xf2, 0x7c, 0x31, 0x35, 0x8c, 0x86, 0xdc, 0xe4, /* 200 */
		0x48, 0xd6, 0x0a, 0x22, 0x3b, 0xdc, 0x55, 0x4e, 0xda, 0xa3,
		0xe4, 0x07, 0x2e, 0xf6, 0x03, 0xa0, 0xf4, 0x61, 0x9f, 0x8d,
		0xb3, 0x9c, 0xec, 0x29, 0x1c, 0x86, 0x01, 0x74, 0x15, 0x5d,
		0x8a, 0xbf, 0x9e, 0x10, 0x82, 0x93, 0x1d, 0xf5, 0x8e, 0xd9,
		0xee, 0x9c, 0x12, 0x15, 0xdd, 0x23, 0x93, 0xde, 0x02, 0xf5,
		0xc1, 0x76, 0x7f, 0x07, 0x0e, 0x28)
};

ke_test_vector_t modp3072 = {
	.method = MODP_3072_BIT,
	.seed = chunk_from_chars(
		0x51, 0xf8, 0xaa, 0xb6, 0x63, 0x20, 0x1e, 0xb2, 0x86, 0xba,
		0xd2, 0x99, 0x32, 0xb2, 0xe5, 0x8a, 0x92, 0x96, 0xbf, 0x2a,
		0xa7, 0x78, 0x79, 0xcc, 0x8c, 0x64, 0x29, 0xd5, 0xa6, 0x68,
		0xad, 0xf7, 0x60, 0x57, 0xad, 0xc3, 0x77, 0xcc, 0x75, 0xfd,
		0x86, 0x47, 0x96, 0xb8, 0xfa, 0x7b, 0x42, 0x8c, /* - */
		0xaf, 0x7f, 0x44, 0x29, 0x0b, 0xaa, 0xc8, 0x81, 0x5e, 0x70,
		0x75, 0x9b, 0x6b, 0x27, 0x42, 0x27, 0x12, 0x22, 0x50, 0xea,
		0x8d, 0x5a, 0x12, 0xef, 0x4b, 0x0a, 0x82, 0xcf, 0xe9, 0x1d,
		0x52, 0x98, 0x9d, 0x96, 0xc1, 0xa8, 0x40, 0x89, 0x2a, 0x9f,
		0xfa, 0x9f, 0x3a, 0x65, 0xc7, 0x7c, 0x8c, 0xd9),
	.pub_i = chunk_from_chars(
		0xae, 0x98, 0x27, 0xa0, 0x7a, 0x37, 0x2e, 0x16, 0x01, 0x67,
		0x20, 0x63, 0x72, 0xc5, 0x5b, 0x61, 0x5c, 0xd4, 0x9e, 0x8c,
		0x43, 0xf7, 0x1a, 0x6e, 0x6b, 0x22, 0x83, 0x8c, 0x0f, 0x9b,
		0x9d, 0x90, 0x3a, 0xe5, 0x97, 0x97, 0xf3, 0x95, 0xc9, 0xca,
		0x7c, 0x78, 0x0f, 0xa8, 0x8d, 0x79, 0x27, 0xa3, 0xac, 0x48,
		0xf1, 0xb5, 0x05, 0xe6, 0x7a, 0x0d, 0xb1, 0xeb, 0xad, 0x57,
		0x8e, 0xb1, 0x57, 0x2b, 0x86, 0x52, 0x5e, 0x84, 0x0c, 0x4c,
		0xe2, 0x43, 0x5e, 0xd4, 0xcd, 0xe1, 0x75, 0x68, 0xa2, 0xcb,
		0x27, 0x08, 0x85, 0xc9, 0xd3, 0xa7, 0x24, 0xf8, 0x73, 0x46,
		0xe5, 0x19, 0xa9, 0x47, 0x84, 0xee, 0x88, 0xaf, 0xec, 0x27, /* 100 */
		0x86, 0x68, 0x9f, 0x0b, 0xc1, 0x23, 0xe0, 0xe4, 0x65, 0x79,
		0x57, 0xb4, 0xf7, 0x5f, 0x24, 0xb5, 0x2d, 0xd4, 0x80, 0x39,
		0x09, 0x3a, 0xe8, 0xeb, 0x5d, 0xa4, 0x0d, 0x1e, 0x31, 0xfc,
		0x53, 0xfd, 0x98, 0xa0, 0x58, 0xb0, 0x38, 0x9b, 0xd5, 0xc7,
		0xaf, 0x5b, 0x0d, 0xf4, 0x49, 0xd6, 0x13, 0x90, 0x1c, 0xa2,
		0x71, 0xb5, 0x8e, 0x47, 0x0f, 0x88, 0xd3, 0x45, 0x93, 0x1f,
		0x3f, 0x64, 0xe8, 0x1f, 0x03, 0x1d, 0xe9, 0x5a, 0xda, 0xef,
		0x78, 0xf6, 0x0e, 0x64, 0x5e, 0xca, 0x7d, 0xd5, 0x61, 0x8a,
		0xa6, 0xed, 0xd3, 0x9f, 0xa8, 0x91, 0x22, 0x46, 0x39, 0xa9,
		0xae, 0x42, 0xa8, 0xa9, 0x37, 0x4d, 0x68, 0x3f, 0xd3, 0xf4, /* 200 */
		0x82, 0x6b, 0x97, 0x4c, 0x8d, 0x00, 0x39, 0x2f, 0x31, 0x12,
		0x45, 0xa7, 0x1c, 0x86, 0xdb, 0x4e, 0xb5, 0xef, 0xc4, 0x78,
		0x34, 0xce, 0x6b, 0xaf, 0xb2, 0x08, 0xaa, 0xba, 0xc6, 0x8c,
		0x35, 0xe3, 0x4f, 0x79, 0x15, 0x78, 0x40, 0x1c, 0x7b, 0x75,
		0x01, 0x15, 0xb1, 0x6e, 0x5c, 0x3a, 0xc3, 0x62, 0x42, 0xf4,
		0xa9, 0x0a, 0x99, 0x73, 0x4c, 0x08, 0x36, 0x8d, 0x48, 0xab,
		0x5d, 0x50, 0xec, 0x92, 0xf3, 0xf1, 0x26, 0x7e, 0x54, 0xb9,
		0x33, 0x34, 0xa5, 0x0b, 0x00, 0xd7, 0x57, 0x1d, 0x1c, 0xeb,
		0x1e, 0x66, 0x21, 0x58, 0xd5, 0x66, 0x36, 0x20, 0xd1, 0xf0,
		0x22, 0x8a, 0x64, 0xf4, 0x92, 0xea, 0x8e, 0x1f, 0xe1, 0x81, /* 300 */
		0x7e, 0x35, 0x83, 0x57, 0x7f, 0x1e, 0x48, 0xa5, 0x84, 0x91,
		0xc1, 0x0d, 0x98, 0xff, 0xa0, 0xea, 0xa8, 0x64, 0x05, 0xd0,
		0x2c, 0x94, 0xb4, 0x9f, 0x99, 0xfd, 0xdc, 0x6a, 0x3b, 0x7b,
		0xd6, 0xe3, 0xbc, 0xe2, 0x26, 0x1e, 0xa3, 0xcb, 0xbd, 0x82,
		0x1f, 0x49, 0x5a, 0x2a, 0xa7, 0x1a, 0xe9, 0x09, 0xb3, 0xcd,
		0x0d, 0x2c, 0x9b, 0xec, 0x0b, 0x06, 0x8b, 0xa7, 0x26, 0x20,
		0xff, 0x06, 0x32, 0x4c, 0xdb, 0x80, 0x0b, 0xf8, 0x56, 0x5e,
		0x4b, 0x78, 0x21, 0x84, 0x61, 0x0e, 0xf0, 0x77, 0xbc, 0x4e,
		0x8f, 0xc5, 0x17, 0xf7),
	.pub_r = chunk_from_chars(
		0x2b, 0x27, 0x0c, 0x8e, 0x66, 0x74, 0x09, 0x31, 0x45, 0xef,
		0x84, 0xab, 0x0c, 0x34, 0x56, 0x0f, 0x5c, 0x4f, 0x94, 0x82,
		0xa5, 0x0f, 0x2f, 0x44, 0x72, 0xfd, 0xe9, 0x87, 0xb4, 0x6e,
		0xb8, 0x40, 0x34, 0x02, 0x03, 0xd6, 0x31, 0x10, 0x2f, 0x2c,
		0x7f, 0x9e, 0xb4, 0x8d, 0x20, 0x74, 0x44, 0xfb, 0x21, 0x71,
		0x7d, 0xaf, 0x76, 0xf6, 0x60, 0x04, 0xa7, 0x9e, 0xa8, 0xee,
		0xe7, 0x7c, 0xc9, 0x80, 0x19, 0xba, 0x12, 0x34, 0xf1, 0x6f,
		0xbf, 0xf7, 0xdd, 0xa9, 0x45, 0xa4, 0x46, 0x81, 0x1d, 0x28,
		0x5d, 0x31, 0xaf, 0xd3, 0x31, 0xdf, 0xa8, 0x0c, 0x87, 0x0e,
		0xfb, 0x19, 0x96, 0x68, 0xbb, 0x38, 0x76, 0x18, 0xe1, 0x16, /* 100 */
		0x21, 0xf5, 0x32, 0x92, 0x90, 0x8f, 0xd3, 0x29, 0xaa, 0xdb,
		0x91, 0x63, 0x5e, 0x47, 0x5f, 0x33, 0x00, 0x4d, 0x9e, 0xff,
		0xca, 0x65, 0xe7, 0x3f, 0xdf, 0xe5, 0x58, 0xf3, 0x4b, 0xa9,
		0xee, 0xf3, 0xcf, 0x96, 0x1d, 0xab, 0x56, 0x5c, 0x5a, 0x72,
		0x2a, 0xfb, 0x22, 0x8c, 0x26, 0xbf, 0x38, 0xd3, 0x9c, 0x84,
		0xf7, 0xca, 0x70, 0xf8, 0x51, 0xbf, 0x21, 0x8a, 0x7e, 0x2c,
		0x2d, 0xb5, 0xd3, 0x56, 0x93, 0x7a, 0x01, 0x5a, 0x65, 0x50,
		0x31, 0x13, 0xf6, 0xc0, 0xd5, 0x2d, 0x3b, 0x2a, 0x21, 0x21,
		0xa0, 0xca, 0x42, 0xc6, 0xcc, 0xde, 0xbb, 0x0b, 0x2c, 0xfc,
		0x68, 0xaa, 0x66, 0x2f, 0x3e, 0x08, 0xdd, 0x69, 0x82, 0x42, /* 200 */
		0x7a, 0x0a, 0x11, 0xb2, 0xe1, 0x44, 0xe0, 0xf1, 0xa3, 0x47,
		0xd1, 0xd2, 0x7f, 0xf8, 0x96, 0xee, 0x56, 0xf5, 0xd9, 0xfb,
		0x25, 0xf7, 0x40, 0xb5, 0xc7, 0x09, 0x88, 0xfc, 0xdc, 0x91,
		0x12, 0xdf, 0xce, 0xa2, 0xde, 0x11, 0x7c, 0x12, 0x87, 0xd3,
		0x2d, 0xa8, 0x42, 0x70, 0x49, 0xce, 0xef, 0x09, 0x7b, 0xf4,
		0x57, 0x81, 0x84, 0x6a, 0x02, 0x1d, 0x38, 0x3e, 0x73, 0x1a,
		0xa6, 0xe5, 0xc3, 0xdc, 0x0a, 0x46, 0x22, 0x6e, 0x54, 0xc5,
		0xa6, 0x36, 0x7a, 0xd8, 0x90, 0x0b, 0x8e, 0xad, 0xba, 0x54,
		0x09, 0x3e, 0xc6, 0xf7, 0x1c, 0x29, 0x74, 0x4a, 0xdc, 0x73,
		0x02, 0xe8, 0x11, 0x85, 0xed, 0x80, 0xdb, 0x14, 0xb3, 0xe4, /* 300 */
		0x3f, 0x29, 0x36, 0x1a, 0xf4, 0xe6, 0x1c, 0x33, 0x1e, 0x5d,
		0x77, 0x4d, 0x76, 0x17, 0xbf, 0x31, 0x55, 0x71, 0x3b, 0x0d,
		0x4f, 0x90, 0x38, 0x97, 0x6f, 0x2f, 0xf8, 0x51, 0x99, 0xc1,
		0xd7, 0x0d, 0xcb, 0x3e, 0xfa, 0x2b, 0x63, 0xe0, 0xc7, 0x7c,
		0xee, 0x78, 0x9f, 0xa0, 0x05, 0x8f, 0xdf, 0x52, 0x67, 0xc6,
		0xb1, 0xbd, 0xed, 0xc1, 0xd1, 0xef, 0x47, 0xca, 0x08, 0x22,
		0x2a, 0x9a, 0x27, 0xe6, 0x5f, 0x89, 0x2e, 0xef, 0xde, 0x06,
		0x12, 0xf4, 0x6b, 0x51, 0xbe, 0x1a, 0x14, 0x77, 0x4d, 0x45,
		0x14, 0xd7, 0xe7, 0xca),
	.shared = chunk_from_chars(
		0x45, 0x3c, 0xaf, 0x89, 0xd6, 0xbf, 0xf3, 0x9b, 0xda, 0xe1,
		0x85, 0x46, 0x17, 0x61, 0xd6, 0xde, 0xf5, 0x89, 0xa5, 0x82,
		0x66, 0x9c, 0x1a, 0xc8, 0x6b, 0x0b, 0x7e, 0xe4, 0x69, 0x56,
		0xc9, 0xf9, 0x4d, 0x74, 0x77, 0xdd, 0xd8, 0x1c, 0xde, 0x12,
		0x75, 0x76, 0xda, 0xeb, 0xd3, 0x52, 0x95, 0x15, 0xf0, 0x6a,
		0xa7, 0xdf, 0x5e, 0xb1, 0x31, 0x27, 0x67, 0x5c, 0xe1, 0xb4,
		0xdc, 0xa2, 0xac, 0xb6, 0x47, 0xe3, 0x55, 0x76, 0xb9, 0x40,
		0x50, 0xbc, 0xc3, 0x85, 0xb3, 0xb4, 0x89, 0x44, 0xdd, 0x1e,
		0xca, 0xbe, 0x6c, 0x76, 0x70, 0x45, 0xcd, 0xcd, 0xdd, 0x3a,
		0x1d, 0x65, 0x83, 0x8c, 0xcc, 0xb0, 0x82, 0xf5, 0x44, 0xf0, /* 100 */
		0xfd, 0x9b, 0xe6, 0xc3, 0xd4, 0xff, 0xe7, 0x55, 0xe4, 0xd5,
		0x6d, 0xfb, 0xb4, 0x20, 0x93, 0x52, 0x3f, 0x45, 0xd2, 0x41,
		0x5e, 0xad, 0xf3, 0x6a, 0x18, 0x5c, 0x87, 0xa6, 0xe0, 0x14,
		0x27, 0xbb, 0xcc, 0x98, 0x95, 0x68, 0xa2, 0x93, 0x66, 0xb1,
		0xc6, 0x91, 0x72, 0x4d, 0x81, 0x0e, 0x99, 0xfb, 0x05, 0x83,
		0x00, 0x26, 0x72, 0xa5, 0x3e, 0x49, 0x9f, 0xf2, 0x30, 0x4a,
		0x4b, 0xcc, 0x10, 0xa3, 0xb8, 0x0b, 0xf4, 0x8a, 0x9e, 0x31,
		0x5b, 0xe9, 0x98, 0xbf, 0x17, 0x0c, 0xe9, 0x0e, 0xc4, 0xeb,
		0x87, 0x05, 0x57, 0x0e, 0x5e, 0xc4, 0x80, 0xc2, 0x1c, 0x4d,
		0xd0, 0xa5, 0x3a, 0x1c, 0xb2, 0x06, 0xa3, 0x42, 0x45, 0x2e, /* 200 */
		0xa8, 0xa6, 0x84, 0xd5, 0x69, 0xd8, 0x10, 0xf8, 0xe9, 0x8f,
		0xea, 0xd7, 0x03, 0x9c, 0x64, 0xef, 0xef, 0x59, 0x35, 0x24,
		0xf5, 0x07, 0x5c, 0xb8, 0x7e, 0x75, 0x63, 0xa4, 0x2a, 0xd1,
		0xb5, 0x78, 0x19, 0xc7, 0xb0, 0x19, 0x96, 0x3d, 0x07, 0x0b,
		0x4c, 0xef, 0xe2, 0x2c, 0x59, 0x6b, 0x3a, 0x96, 0xa8, 0xba,
		0x41, 0xbd, 0xab, 0xe0, 0x54, 0xa9, 0xa9, 0xb4, 0xd1, 0xc4,
		0x27, 0x8a, 0x83, 0x64, 0xea, 0xd9, 0x96, 0xa0, 0x39, 0xe8,
		0xbd, 0xdc, 0x00, 0x60, 0x1e, 0xcf, 0x10, 0x2d, 0x20, 0xd6,
		0xab, 0xce, 0xd2, 0xff, 0x19, 0x29, 0x49, 0xf5, 0xf8, 0x1c,
		0x36, 0xfb, 0x47, 0x96, 0x44, 0x39, 0xcd, 0x44, 0x03, 0x6a, /* 300 */
		0x23, 0xac, 0x30, 0x86, 0x8c, 0xa5, 0x23, 0x25, 0x84, 0x7b,
		0x31, 0x00, 0xd3, 0xd4, 0x28, 0x16, 0x76, 0xbf, 0x94, 0x84,
		0xa0, 0x05, 0x66, 0xf1, 0x3b, 0x6a, 0xe8, 0x9c, 0x57, 0xb7,
		0x73, 0x2f, 0x8e, 0x60, 0x43, 0x4e, 0x8e, 0x48, 0x85, 0xcb,
		0x1b, 0xf9, 0x47, 0xce, 0xd1, 0x4d, 0x31, 0x53, 0xe1, 0xb6,
		0xc8, 0x1b, 0x33, 0x12, 0x0f, 0xfb, 0xd8, 0x45, 0x94, 0x91,
		0xd2, 0xc5, 0x78, 0xaa, 0xb0, 0xa9, 0x7e, 0x83, 0xdc, 0x87,
		0x87, 0x5b, 0xe5, 0x88, 0xc0, 0xcd, 0xee, 0xee, 0xfd, 0x19,
		0xcc, 0x4f, 0x1d, 0x40)
};

ke_test_vector_t modp4096 = {
	.method = MODP_4096_BIT,
	.seed = chunk_from_chars(
		0xab, 0x69, 0xbc, 0xe9, 0x61, 0xf9, 0x8a, 0xa9, 0xd5, 0x91,
		0xe3, 0xfd, 0x9a, 0xbc, 0x46, 0xc8, 0x0d, 0xde, 0x39, 0x02,
		0x84, 0xf1, 0x91, 0x42, 0xe8, 0x81, 0x5a, 0xb0, 0x80, 0x54,
		0x72, 0x2b, 0xbd, 0x2e, 0x14, 0x1e, 0x27, 0x9e, 0xc7, 0xfd,
		0x30, 0xaa, 0xfa, 0xca, 0x66, 0x40, 0x93, 0x73, 0x1e, 0xcc,
		0x75, 0xa2, 0xbd, 0x07, 0xe4, 0xa5, 0x88, 0x3f, 0x56, 0x08,
		0x93, 0xc8, 0x33, 0x50, /* - */
		0xef, 0x3e, 0xcb, 0x88, 0xf4, 0xf9, 0x88, 0xf7, 0x2a, 0xcd,
		0x5a, 0xa2, 0x2e, 0xbe, 0xbf, 0x19, 0x0f, 0xde, 0x7e, 0x8e,
		0x4d, 0x0d, 0x50, 0xcc, 0xde, 0x80, 0xf9, 0x41, 0xb3, 0xbb,
		0xd6, 0xa2, 0xf9, 0xa1, 0x2a, 0xee, 0x9b, 0xb3, 0xe5, 0xc5,
		0x80, 0x3f, 0x67, 0xaf, 0xfe, 0x27, 0xc8, 0x0d, 0x05, 0x46,
		0x9b, 0x52, 0x54, 0xeb, 0x1f, 0x32, 0x6c, 0x18, 0x73, 0x1f,
		0xd3, 0x4a, 0xc2, 0xcd),
	.pub_i = chunk_from_chars(
		0x3e, 0xc6, 0x3c, 0x0c, 0x68, 0x32, 0x05, 0xc3, 0x9d, 0x4b,
		0x97, 0x72, 0x39, 0xfe, 0x7f, 0x96, 0x17, 0x56, 0x0b, 0x1a,
		0x56, 0xc4, 0x7f, 0xd9, 0x07, 0x02, 0x6a, 0xd8, 0x09, 0x0a,
		0xa4, 0x4b, 0xcc, 0x29, 0xd3, 0xc2, 0xb3, 0x8d, 0x34, 0xe1,
		0x5c, 0x9e, 0x14, 0xc8, 0x05, 0x81, 0xa5, 0xc4, 0x83, 0x2f,
		0x5c, 0xd8, 0x8e, 0x43, 0x73, 0xbf, 0x59, 0x0e, 0xad, 0x63,
		0x6f, 0x68, 0xc8, 0x99, 0x59, 0xdf, 0xeb, 0xc4, 0x33, 0x03,
		0x48, 0x97, 0x4e, 0xc4, 0x80, 0x7c, 0xa4, 0x4f, 0xb7, 0xd4,
		0x4f, 0x02, 0xb7, 0x97, 0x70, 0x9e, 0x16, 0x0e, 0x3c, 0x0e,
		0xc4, 0x06, 0x1f, 0x75, 0x9a, 0xec, 0x63, 0xd9, 0x96, 0xd2, /* 100 */
		0x37, 0x7a, 0x18, 0xdc, 0xc5, 0x94, 0xc2, 0x08, 0xfa, 0x83,
		0x7a, 0xeb, 0xc1, 0x68, 0x9d, 0x5c, 0x0a, 0xb4, 0x30, 0x5c,
		0x1b, 0xbe, 0x86, 0xb1, 0xd4, 0xa1, 0x6f, 0x4c, 0xb5, 0x25,
		0xfc, 0xcc, 0xf1, 0x00, 0x6e, 0x21, 0x23, 0xc7, 0x76, 0xb1,
		0xff, 0xd1, 0xfe, 0xa2, 0x97, 0x7b, 0x1e, 0xac, 0x82, 0xd1,
		0xee, 0xec, 0x4c, 0x46, 0x73, 0xa5, 0x17, 0xdb, 0xc4, 0x2e,
		0x0f, 0x89, 0x30, 0xdb, 0x28, 0xd8, 0xc8, 0xe8, 0xb4, 0x0f,
		0x8a, 0x49, 0x8d, 0xa5, 0x83, 0x05, 0x5c, 0x9c, 0x12, 0x35,
		0x34, 0xcc, 0x2c, 0x53, 0x34, 0xd0, 0xbe, 0xbe, 0xa3, 0x76,
		0x26, 0x78, 0xd1, 0xf5, 0x34, 0xba, 0x64, 0x6f, 0x75, 0x22, /* 200 */
		0xf2, 0x68, 0x57, 0xa0, 0xff, 0x28, 0x8f, 0x7e, 0xfc, 0x38,
		0xc1, 0xdf, 0xa0, 0xa6, 0x4c, 0x3e, 0xb5, 0x31, 0x64, 0xc1,
		0x8f, 0x6c, 0x2a, 0x4e, 0x51, 0x5c, 0x13, 0xc1, 0xb7, 0x39,
		0x35, 0xd9, 0xbd, 0xb4, 0x58, 0x27, 0x33, 0x55, 0x34, 0x29,
		0x67, 0xa8, 0x54, 0xc7, 0x8b, 0x16, 0x8b, 0x58, 0x90, 0xf7,
		0xfe, 0x08, 0xd0, 0x8c, 0x2e, 0x4a, 0x6f, 0x16, 0x63, 0x32,
		0x2e, 0x7b, 0x52, 0xef, 0x02, 0x17, 0xc8, 0x15, 0x13, 0x72,
		0x2f, 0x34, 0xc0, 0x95, 0x48, 0x02, 0xe4, 0x2c, 0xfb, 0x6c,
		0x9e, 0xe9, 0xd2, 0xc3, 0x98, 0x68, 0x71, 0x1f, 0x1a, 0xfc,
		0x4e, 0x47, 0x78, 0x52, 0x2d, 0x9c, 0x45, 0x09, 0x0b, 0x26, /* 300 */
		0x23, 0xf4, 0xaf, 0xa3, 0x88, 0x42, 0x6b, 0x89, 0x75, 0x64,
		0x05, 0x93, 0xbe, 0x79, 0x2b, 0xb8, 0xa7, 0x15, 0xbe, 0xf2,
		0xc4, 0x6c, 0x2c, 0x50, 0xa6, 0x8a, 0xa6, 0x30, 0x13, 0x8a,
		0xb2, 0x6d, 0xf9, 0xda, 0xe2, 0xb0, 0xe2, 0xc4, 0xeb, 0xc1,
		0x0a, 0x86, 0x36, 0xda, 0xd9, 0xbe, 0x2f, 0xfb, 0x79, 0x19,
		0xcd, 0x52, 0x78, 0x85, 0x84, 0x3c, 0xb7, 0x6c, 0x71, 0xf2,
		0x8e, 0x71, 0xf6, 0xf8, 0x23, 0xd9, 0x9c, 0x2f, 0xe3, 0x1a,
		0xd7, 0xfb, 0xcc, 0x9a, 0xbb, 0x24, 0xef, 0x26, 0x66, 0xa1,
		0x21, 0xcc, 0xf2, 0x1f, 0x88, 0x1f, 0xa6, 0x6a, 0xb7, 0x18,
		0xba, 0xf9, 0x2b, 0xef, 0x5c, 0xec, 0xaa, 0xcf, 0x4c, 0x92, /* 400 */
		0x9c, 0x9c, 0xb5, 0x44, 0x24, 0xd1, 0x7a, 0xb7, 0x73, 0x8c,
		0xa6, 0x80, 0x7b, 0x4e, 0xfa, 0x49, 0x36, 0x0a, 0x3d, 0xb6,
		0x52, 0x7c, 0x72, 0x4c, 0x45, 0x31, 0x62, 0x9e, 0x9d, 0xb1,
		0x84, 0x5c, 0x6a, 0x1e, 0xb4, 0x63, 0x2e, 0x0c, 0x07, 0xab,
		0x6c, 0x31, 0x37, 0x7c, 0x80, 0x83, 0x70, 0x49, 0x46, 0x44,
		0xa3, 0x5b, 0xd4, 0x78, 0x5f, 0x49, 0x5f, 0x60, 0x63, 0x8b,
		0x98, 0x81, 0x71, 0xde, 0xce, 0x5d, 0x5c, 0x44, 0x90, 0x35,
		0x7d, 0x02, 0x89, 0xb3, 0x94, 0x4b, 0xc9, 0xf7, 0x85, 0x4d,
		0xb8, 0x06, 0x97, 0xd5, 0xf6, 0x3b, 0xf3, 0xe3, 0x1d, 0xb4,
		0x4e, 0xfb, 0x9a, 0x43, 0xe1, 0xdc, 0x9d, 0x17, 0xee, 0x47, /* 500 */
		0x01, 0x2d, 0xfb, 0xac, 0xfd, 0xaf, 0x4d, 0xeb, 0xfe, 0xf0,
		0x8b, 0x5c),
	.pub_r = chunk_from_chars(
		0x20, 0xb7, 0xb4, 0x5f, 0x69, 0xe2, 0xc3, 0x0b, 0xcc, 0xf8,
		0x41, 0xca, 0xe8, 0x04, 0x2c, 0x36, 0x35, 0x37, 0x25, 0x42,
		0x05, 0x99, 0x33, 0x5f, 0xa2, 0xe8, 0x7b, 0xbe, 0x59, 0xce,
		0xbb, 0xc7, 0xa7, 0xd3, 0xf6, 0x1e, 0xb6, 0x69, 0x82, 0x50,
		0x3a, 0x75, 0x76, 0xc3, 0x47, 0x63, 0xdf, 0xef, 0x6a, 0x6b,
		0x18, 0x0a, 0x93, 0xaf, 0x66, 0xe4, 0xf7, 0x2f, 0x12, 0xd5,
		0x8c, 0x93, 0x84, 0x6d, 0x16, 0x05, 0x58, 0xb0, 0xd3, 0x16,
		0x03, 0x9f, 0x6b, 0xa9, 0x9e, 0xa6, 0x4f, 0x00, 0x5b, 0xa1,
		0x1e, 0x59, 0xf3, 0xa9, 0xcb, 0x99, 0x3d, 0x28, 0x27, 0x1a,
		0x4f, 0xb8, 0x30, 0xc3, 0xf6, 0xc4, 0xce, 0xb9, 0xb0, 0x16, /* 100 */
		0x2c, 0xcc, 0xa1, 0x97, 0xff, 0x65, 0x15, 0x78, 0x9d, 0x43,
		0x6c, 0x94, 0x7e, 0xb5, 0xd8, 0x01, 0x09, 0x74, 0xeb, 0xcd,
		0x36, 0x6b, 0xc4, 0x76, 0x83, 0x41, 0x09, 0x0a, 0x6f, 0xb5,
		0x5c, 0xa8, 0x4e, 0x31, 0xd2, 0x48, 0x9e, 0x35, 0x27, 0xa2,
		0x60, 0x77, 0x6f, 0x9b, 0x8a, 0x58, 0x57, 0x7b, 0xdc, 0xd6,
		0x89, 0xd8, 0xe4, 0xb7, 0x25, 0x14, 0xcf, 0x15, 0xee, 0xa5,
		0xa4, 0x96, 0x29, 0xa2, 0xf2, 0xc4, 0x86, 0xc5, 0x1b, 0x5d,
		0x14, 0xd4, 0x9b, 0x11, 0x93, 0x09, 0xbf, 0xe9, 0xc9, 0x32,
		0xb6, 0x04, 0xc6, 0xf1, 0xc0, 0xe9, 0x2c, 0x44, 0x8d, 0xc1,
		0x9f, 0x54, 0xf4, 0x21, 0x11, 0x2f, 0x28, 0x87, 0x23, 0x8c, /* 200 */
		0x91, 0x37, 0xc5, 0x59, 0xb6, 0x9a, 0x93, 0xac, 0xf3, 0x6c,
		0xc1, 0xf0, 0xbd, 0xfe, 0x4c, 0xca, 0x0b, 0x60, 0x47, 0x71,
		0xee, 0x2a, 0xf1, 0x7c, 0x34, 0x04, 0x5d, 0x42, 0x29, 0xb2,
		0xb8, 0x0a, 0xcd, 0xfb, 0x7f, 0x56, 0xe4, 0xea, 0xee, 0x81,
		0xed, 0x6c, 0x88, 0x5a, 0x2e, 0x45, 0xaf, 0xc9, 0x8d, 0xe1,
		0x21, 0x2e, 0x5f, 0x71, 0x4f, 0x5f, 0x00, 0x88, 0x12, 0xd7,
		0x17, 0x06, 0x89, 0x6d, 0x2d, 0x5e, 0xd8, 0x59, 0x0c, 0xee,
		0x66, 0x41, 0x08, 0x79, 0xdc, 0x31, 0x95, 0xa9, 0x21, 0xef,
		0xe0, 0x85, 0xdb, 0x41, 0xd6, 0x87, 0xec, 0x2d, 0xe9, 0x06,
		0xa8, 0x10, 0x33, 0x6f, 0xa7, 0x57, 0x0d, 0x43, 0x11, 0x2d, /* 300 */
		0xec, 0xf9, 0xff, 0xa4, 0xae, 0x50, 0xf5, 0x4f, 0xad, 0x3e,
		0xec, 0x2b, 0xb3, 0xed, 0x86, 0xdd, 0xa3, 0x66, 0x2b, 0xc0,
		0xfc, 0x1c, 0x28, 0x94, 0xd4, 0xde, 0x7d, 0xa9, 0x26, 0x0f,
		0x74, 0x73, 0xbe, 0x67, 0x6d, 0xbf, 0x60, 0x90, 0x33, 0x32,
		0x0d, 0xba, 0xa4, 0x91, 0x72, 0x0c, 0xe2, 0xd3, 0x5d, 0x07,
		0x8d, 0xbd, 0xde, 0x84, 0x5b, 0x4a, 0x49, 0x5a, 0xd2, 0xec,
		0xc2, 0xe6, 0xda, 0x5f, 0x1c, 0x0c, 0x89, 0x20, 0xe4, 0xea,
		0x0e, 0xcc, 0xa0, 0xe0, 0x77, 0xda, 0xc5, 0x9b, 0x97, 0xf0,
		0xe9, 0x4b, 0x6c, 0xca, 0xd3, 0xf4, 0x2a, 0x34, 0xb6, 0x63,
		0x4e, 0xde, 0x83, 0xb2, 0x9d, 0x9c, 0x76, 0x15, 0xf7, 0x41, /* 400 */
		0x90, 0xdf, 0x33, 0xb2, 0x0e, 0x7d, 0x69, 0x08, 0xd6, 0x63,
		0x69, 0x8a, 0xdb, 0x8a, 0x2d, 0xac, 0xd9, 0x79, 0xf1, 0xca,
		0x13, 0xf2, 0xcc, 0x84, 0x02, 0xf8, 0xf6, 0xcb, 0xf9, 0x11,
		0x92, 0xf1, 0x11, 0xdd, 0xf5, 0xad, 0x29, 0x39, 0x05, 0x95,
		0x54, 0xac, 0x47, 0xb4, 0x78, 0x74, 0x7b, 0x78, 0xa2, 0x71,
		0x92, 0x97, 0xae, 0xda, 0x20, 0xe3, 0xd8, 0x53, 0x38, 0x26,
		0x9b, 0x30, 0x80, 0xfc, 0x3f, 0xd7, 0x8d, 0xe4, 0xac, 0xf3,
		0x4e, 0x09, 0x47, 0x7a, 0x61, 0xca, 0xc7, 0xc1, 0xb7, 0xdc,
		0x76, 0x94, 0xed, 0x14, 0x1a, 0x51, 0x48, 0x0e, 0x6b, 0x2b,
		0x43, 0xc0, 0x25, 0x56, 0xe8, 0xa8, 0x49, 0x7c, 0x4f, 0xca, /* 500 */
		0x56, 0xf7, 0xfd, 0x56, 0xc8, 0xeb, 0x36, 0xa1, 0xf8, 0xc3,
		0xd1, 0x24),
	.shared = chunk_from_chars(
		0x23, 0xb0, 0x6f, 0x49, 0x39, 0x60, 0x9b, 0x0f, 0x67, 0x08,
		0x85, 0x2d, 0x4f, 0x87, 0xb3, 0x56, 0x98, 0x8f, 0xb4, 0x5b,
		0x1e, 0x4c, 0x95, 0x1b, 0x0b, 0x1d, 0x59, 0xbb, 0xa8, 0xca,
		0x8b, 0x60, 0xc2, 0x55, 0xa1, 0x8f, 0x37, 0x55, 0xa6, 0xdb,
		0x05, 0xe4, 0x28, 0x5f, 0xe8, 0xf1, 0xf9, 0xd2, 0x6a, 0xdd,
		0x24, 0x14, 0x19, 0xf0, 0xfc, 0xa9, 0x82, 0x37, 0xfd, 0x7a,
		0x5e, 0x52, 0x6d, 0x57, 0xab, 0x5a, 0xd0, 0x69, 0xc3, 0xf1,
		0x89, 0xf3, 0xc2, 0x91, 0x50, 0x11, 0x44, 0xa6, 0x55, 0x14,
		0xa5, 0x66, 0xb7, 0x4d, 0x0d, 0xc6, 0x41, 0x9c, 0x13, 0x06,
		0xcf, 0xbf, 0xf3, 0x5d, 0xbe, 0xb4, 0x3a, 0xbd, 0x5e, 0xd3, /* 100 */
		0x9c, 0xe1, 0x37, 0x53, 0xb2, 0x8a, 0xe5, 0xe1, 0x05, 0xf7,
		0x19, 0x1b, 0xc6, 0xd7, 0x0a, 0xc3, 0x55, 0x2c, 0x82, 0x37,
		0x3c, 0x6b, 0xd9, 0xdb, 0x8e, 0xd5, 0xee, 0x15, 0x46, 0xfd,
		0xb5, 0x49, 0x9a, 0xe7, 0x0a, 0x6b, 0xcb, 0xd1, 0x85, 0x9a,
		0x31, 0xec, 0x43, 0xdc, 0xcf, 0x05, 0x52, 0x5e, 0xe4, 0x0c,
		0x94, 0x98, 0x87, 0xdd, 0x81, 0x68, 0xae, 0x29, 0xcc, 0x53,
		0x7a, 0x6f, 0x57, 0xa5, 0x26, 0xf4, 0x25, 0xdd, 0x07, 0x5f,
		0x39, 0xd6, 0xee, 0x71, 0xcb, 0x49, 0x7a, 0x57, 0x1f, 0xe5,
		0x79, 0x58, 0xc7, 0xde, 0x32, 0x1d, 0x64, 0xf4, 0xe4, 0x89,
		0x22, 0x43, 0xe7, 0x8f, 0xef, 0x47, 0xb3, 0x31, 0xa6, 0x76, /* 200 */
		0x84, 0x49, 0x89, 0x19, 0x1f, 0x97, 0xad, 0xf7, 0x91, 0x32,
		0x60, 0x7a, 0x14, 0x8f, 0x19, 0x3c, 0x7d, 0xd5, 0xe6, 0xd5,
		0x99, 0x25, 0x7e, 0x1b, 0xf1, 0x21, 0x93, 0x24, 0x68, 0xdb,
		0xbe, 0x21, 0x60, 0xc9, 0x7a, 0xf0, 0x3f, 0x9d, 0x1b, 0x19,
		0xb1, 0x6b, 0x4f, 0x8f, 0xec, 0x5d, 0xcb, 0xc7, 0x98, 0x34,
		0x4a, 0x87, 0xdb, 0xd4, 0x02, 0xa6, 0x26, 0x6e, 0x10, 0xc0,
		0xe8, 0xa7, 0x22, 0xfe, 0x9f, 0x67, 0xe8, 0x63, 0x6c, 0xb0,
		0xa7, 0x3e, 0x22, 0x4d, 0x53, 0x23, 0xde, 0x9b, 0x7f, 0xa7,
		0xc6, 0x6c, 0x62, 0xa1, 0xf4, 0x00, 0x42, 0x04, 0x1c, 0xba,
		0xde, 0xf2, 0x4b, 0x4f, 0xaa, 0xfd, 0xa9, 0x14, 0x79, 0xec, /* 300 */
		0x91, 0x97, 0x64, 0xb0, 0xf4, 0x8b, 0x95, 0x9e, 0x67, 0x99,
		0xf0, 0x94, 0x96, 0x6d, 0x24, 0x61, 0x27, 0xc0, 0x0e, 0x9c,
		0xc7, 0xd8, 0xf5, 0xa8, 0x43, 0xd1, 0xa4, 0xd6, 0x1c, 0x5c,
		0x0a, 0x64, 0xb6, 0xb1, 0x6c, 0xa7, 0x32, 0x44, 0x93, 0x75,
		0xd9, 0xcf, 0x5d, 0x32, 0xd3, 0x99, 0xf6, 0x56, 0xfd, 0x51,
		0x4f, 0xbf, 0xe6, 0x6e, 0xea, 0x82, 0xe4, 0x79, 0xfc, 0x73,
		0x18, 0x0b, 0x72, 0x1d, 0xd4, 0xc5, 0xbb, 0x20, 0xd4, 0x50,
		0xc5, 0xa1, 0x95, 0x9e, 0x1f, 0x8f, 0xed, 0x9d, 0xd2, 0x8c,
		0x97, 0x05, 0x12, 0x72, 0xf6, 0x64, 0x00, 0xcd, 0xd4, 0x13,
		0x45, 0x7d, 0xdd, 0xe6, 0x9a, 0xc7, 0x43, 0x5e, 0xe4, 0xa4, /* 400 */
		0xf7, 0x2a, 0x37, 0x02, 0x49, 0x82, 0xb4, 0xa7, 0xf6, 0xf5,
		0x5e, 0x03, 0x07, 0x03, 0x82, 0xb8, 0x3b, 0x2c, 0xed, 0xb7,
		0x75, 0x25, 0x17, 0xf0, 0x48, 0xb7, 0xc6, 0x91, 0xd1, 0xf1,
		0xd7, 0xb8, 0x52, 0xa5, 0xb7, 0xcd, 0x3b, 0x2b, 0xde, 0x97,
		0x62, 0x0e, 0x9b, 0x2c, 0xd9, 0xc7, 0x7f, 0xd8, 0xcc, 0xb6,
		0x92, 0x5a, 0xde, 0xf7, 0x06, 0xa6, 0x77, 0x0a, 0x2e, 0xfb,
		0x62, 0x1c, 0x93, 0xf1, 0xca, 0x24, 0xf0, 0x9a, 0x68, 0x6e,
		0x8b, 0x86, 0x05, 0x81, 0x49, 0x47, 0x39, 0x92, 0x15, 0x33,
		0x9a, 0x1f, 0x29, 0xfb, 0x57, 0xac, 0xf9, 0xce, 0x9e, 0xba,
		0x2c, 0xbd, 0x49, 0x69, 0xc8, 0x9e, 0x4f, 0xb9, 0x39, 0x02, /* 500 */
		0x12, 0xb9, 0xb2, 0xa3, 0x5d, 0x4a, 0xfa, 0x17, 0xb3, 0xee,
		0xc0, 0x8a)
};

ke_test_vector_t modp6144 = {
	.method = MODP_6144_BIT,
	.seed = chunk_from_chars(
		0xab, 0x36, 0xf0, 0x65, 0x7c, 0x4f, 0xba, 0xdc, 0x2a, 0x3b,
		0x07, 0xed, 0xd1, 0xc8, 0xaf, 0xcb, 0x42, 0xaf, 0xcd, 0x7f,
		0xf9, 0x1c, 0x57, 0x01, 0x37, 0x25, 0x50, 0x0d, 0x89, 0x42,
		0x9f, 0x34, 0x79, 0x8f, 0x99, 0xf5, 0xde, 0x53, 0xd1, 0x08,
		0x8f, 0xd9, 0xf6, 0x60, 0xa1, 0xa5, 0x2b, 0xe4, 0x54, 0xf9,
		0x63, 0x4d, 0x15, 0xbb, 0x4e, 0x4c, 0xe0, 0x9e, 0x5a, 0x69,
		0xe0, 0xf9, 0x9d, 0x59, /* - */
		0x59, 0xa5, 0x52, 0xa4, 0x0d, 0x17, 0x80, 0xaf, 0x64, 0x33,
		0xbc, 0x9e, 0x8a, 0x9b, 0x6d, 0x48, 0x30, 0xdd, 0xd3, 0x57,
		0x53, 0x07, 0x40, 0x9a, 0x90, 0xc9, 0x2b, 0xda, 0x40, 0x4b,
		0xb7, 0x99, 0xf0, 0x09, 0x3e, 0xfe, 0x21, 0xbe, 0xad, 0x85,
		0xf1, 0x6a, 0x23, 0x7b, 0x05, 0x90, 0xc3, 0x35, 0x6d, 0x96,
		0x8c, 0xc0, 0x9d, 0xcd, 0x5b, 0x2d, 0x96, 0x86, 0x07, 0xd6,
		0xd1, 0xbf, 0x70, 0xc2),
	.pub_i = chunk_from_chars(
		0x0a, 0x78, 0x01, 0x5c, 0x3e, 0x0d, 0x36, 0x2a, 0xff, 0xe5,
		0xd9, 0x3a, 0x9c, 0x2d, 0xf2, 0xdf, 0x5b, 0xb9, 0x6b, 0x7f,
		0xf2, 0xfc, 0xc2, 0xdd, 0x96, 0x6c, 0x53, 0x9d, 0x1d, 0x4c,
		0xdb, 0xac, 0xec, 0xe2, 0x6d, 0x16, 0xab, 0x6d, 0x3d, 0xe8,
		0x24, 0xe0, 0x75, 0x87, 0x29, 0x12, 0xeb, 0xa0, 0x44, 0x33,
		0x66, 0x11, 0x0e, 0x0d, 0x7e, 0x17, 0x27, 0x4b, 0x95, 0x78,
		0xaf, 0x12, 0x46, 0x63, 0xe6, 0x55, 0x17, 0x0b, 0xf7, 0xb7,
		0xb2, 0xbd, 0xb4, 0xbf, 0xe4, 0x7a, 0x88, 0x97, 0x68, 0x95,
		0x6a, 0x98, 0xf1, 0xce, 0x1a, 0xc8, 0xc6, 0xc7, 0xc4, 0x8e,
		0x13, 0xa3, 0x6c, 0x16, 0x1c, 0x82, 0x00, 0x7c, 0x98, 0x35, /* 100 */
		0xb2, 0x2e, 0x7b, 0x47, 0x99, 0x38, 0x92, 0xa1, 0x71, 0x2b,
		0x27, 0x5d, 0xdc, 0x84, 0x54, 0xf4, 0x33, 0x0b, 0xd1, 0xfc,
		0x08, 0xab, 0xfc, 0x48, 0x8e, 0x71, 0x8b, 0x43, 0xa6, 0x21,
		0x3a, 0x09, 0xc5, 0x68, 0x74, 0xce, 0xef, 0xb3, 0xa9, 0xfa,
		0xe7, 0xe1, 0x11, 0xc0, 0x0a, 0x10, 0x43, 0x3f, 0x4b, 0x23,
		0x75, 0xef, 0xab, 0x04, 0x2b, 0xd5, 0xc1, 0xc2, 0x9a, 0xaf,
		0x97, 0x0c, 0xeb, 0xae, 0xb5, 0x7f, 0x10, 0xf0, 0x10, 0x6e,
		0xa6, 0x3d, 0x25, 0x72, 0x59, 0x93, 0x0a, 0xf5, 0xb3, 0x3b,
		0xc2, 0x64, 0x27, 0xe6, 0xef, 0x47, 0x32, 0xde, 0x43, 0xdc,
		0xea, 0x0a, 0x88, 0x72, 0x9b, 0x93, 0x7e, 0x6a, 0x9a, 0xfc, /* 200 */
		0xf2, 0x92, 0xa2, 0x9f, 0x8b, 0xe9, 0x9b, 0x71, 0x88, 0xd8,
		0x2d, 0xfc, 0x13, 0x7e, 0x69, 0x19, 0x9a, 0x53, 0x5d, 0x92,
		0xdc, 0x61, 0x37, 0x60, 0x03, 0x38, 0x67, 0x1e, 0x6d, 0x46,
		0x76, 0x31, 0xff, 0xc2, 0xbd, 0x69, 0x59, 0x42, 0xcd, 0x7e,
		0xbb, 0x40, 0x2c, 0x54, 0xea, 0x39, 0x1b, 0x9a, 0x02, 0xde,
		0x1f, 0x28, 0x99, 0xfe, 0xd5, 0xbd, 0xb7, 0x1f, 0x27, 0xab,
		0x71, 0x0b, 0xba, 0xaa, 0x32, 0xb2, 0xc2, 0xe0, 0xec, 0x8d,
		0x4b, 0xd4, 0xca, 0x6c, 0xc5, 0x07, 0xd9, 0x72, 0x0b, 0x20,
		0xaf, 0x9c, 0xce, 0xcd, 0x7f, 0x4e, 0x88, 0xed, 0x55, 0x0a,
		0xea, 0xbc, 0x43, 0xdd, 0x0b, 0x3d, 0xc0, 0x20, 0xdb, 0x3e, /* 300 */
		0x14, 0x89, 0x76, 0xc7, 0x61, 0xf5, 0x44, 0x21, 0x8a, 0x79,
		0xb7, 0x3b, 0x37, 0x77, 0x24, 0x99, 0xf3, 0x61, 0xba, 0x0b,
		0x1d, 0x3c, 0xf2, 0x10, 0x23, 0x75, 0x36, 0xfb, 0x89, 0xd8,
		0x57, 0x86, 0x51, 0xed, 0x67, 0x51, 0xd1, 0xe8, 0x10, 0x95,
		0x61, 0x2b, 0x0f, 0xcf, 0x65, 0x36, 0xbc, 0xb0, 0xff, 0x17,
		0x2c, 0x3d, 0x54, 0xdc, 0x07, 0x13, 0x19, 0x99, 0xd4, 0x11,
		0x98, 0xf9, 0x7e, 0xa8, 0x32, 0x9a, 0xbb, 0x04, 0xc3, 0x75,
		0x3f, 0x83, 0xe1, 0xfd, 0x3b, 0x92, 0x78, 0x72, 0x3c, 0x98,
		0x67, 0xf4, 0xc1, 0xff, 0x19, 0xe1, 0xd2, 0xad, 0x7d, 0x34,
		0x65, 0xf0, 0xb8, 0xc2, 0xdd, 0x9d, 0x4c, 0xcd, 0x36, 0x1a, /* 400 */
		0xbd, 0xf8, 0x56, 0x66, 0xd6, 0xfe, 0x04, 0x2c, 0x98, 0x04,
		0x2b, 0xec, 0xa9, 0x4b, 0x66, 0x4b, 0x71, 0xcf, 0x78, 0x07,
		0x56, 0xe5, 0xba, 0x9c, 0x8a, 0x72, 0xb8, 0xc9, 0xe4, 0x82,
		0xd6, 0x59, 0x22, 0x59, 0x39, 0x75, 0xd6, 0xdd, 0x00, 0xf3,
		0x16, 0xc7, 0xb2, 0x0c, 0x81, 0xeb, 0x67, 0x4f, 0x0b, 0xbe,
		0xa8, 0x1e, 0xed, 0xe6, 0x7b, 0xbf, 0xf1, 0x17, 0x38, 0x3f,
		0xf4, 0x86, 0x0b, 0x97, 0x75, 0xa7, 0x81, 0x86, 0x14, 0xb8,
		0x6d, 0x48, 0x5e, 0x88, 0x98, 0xa9, 0x2f, 0x54, 0xfd, 0x7f,
		0x05, 0x45, 0xb4, 0x32, 0xcd, 0x5f, 0xab, 0xec, 0x2e, 0xa8,
		0x42, 0xd8, 0x3b, 0x51, 0xc2, 0x18, 0x91, 0x7a, 0xb6, 0x10, /* 500 */
		0x5e, 0x26, 0x8b, 0xc8, 0x50, 0x08, 0x2c, 0x79, 0xa1, 0xd0,
		0x74, 0x12, 0xce, 0xc4, 0xd2, 0x3c, 0xb0, 0x04, 0x54, 0xa8,
		0x87, 0x2b, 0x9f, 0xb3, 0x05, 0x4a, 0x89, 0x64, 0xb5, 0xaf,
		0x7f, 0x64, 0x4b, 0x45, 0xcd, 0xd7, 0xf2, 0xb8, 0xa8, 0x8c,
		0xd8, 0x7c, 0x6e, 0xe6, 0x19, 0xd9, 0xaf, 0x59, 0xb5, 0x2b,
		0x07, 0x37, 0x1b, 0x8f, 0xb0, 0x57, 0xe7, 0x05, 0x9f, 0x21,
		0x52, 0x6d, 0xc4, 0x89, 0x4a, 0xff, 0xce, 0xda, 0xc8, 0x5b,
		0x73, 0xf3, 0xd4, 0x07, 0xc7, 0x29, 0x02, 0x7e, 0xa6, 0x79,
		0x82, 0xd3, 0x01, 0xba, 0x93, 0x0e, 0x37, 0x17, 0x3d, 0xfc,
		0x38, 0xd3, 0x25, 0x7e, 0x52, 0xd2, 0x53, 0xba, 0x20, 0xe8, /* 600 */
		0xe9, 0xef, 0xa2, 0x96, 0x38, 0x49, 0x14, 0xd2, 0x83, 0x8b,
		0x2c, 0x62, 0xb0, 0x27, 0xc6, 0x5d, 0x36, 0x34, 0xd4, 0x58,
		0x14, 0x25, 0x6e, 0xc1, 0xcf, 0xd0, 0x2d, 0x21, 0xa3, 0xc0,
		0x9c, 0x9b, 0x14, 0x20, 0x83, 0xec, 0x1a, 0xeb, 0x14, 0x2a,
		0xd3, 0x97, 0x40, 0xad, 0xd0, 0xeb, 0x51, 0x8f, 0xa2, 0x10,
		0x62, 0xb4, 0x50, 0x94, 0xff, 0x35, 0x43, 0xc2, 0x29, 0x88,
		0x0e, 0xf6, 0xb9, 0x4c, 0x85, 0x80, 0x13, 0xed, 0x2f, 0x56,
		0x15, 0xdc, 0x0f, 0x09, 0xd2, 0xe5, 0x40, 0x11, 0x70, 0x34,
		0x76, 0x2c, 0xed, 0xb1, 0xac, 0xe5, 0x82, 0x77, 0x45, 0x42,
		0x3e, 0x8e, 0x8d, 0x08, 0x6e, 0x5b, 0xbe, 0x34, 0xf9, 0x93, /* 700 */
		0x0f, 0x8a, 0x43, 0xec, 0xa6, 0x9f, 0x7c, 0x56, 0xe6, 0x95,
		0x31, 0x85, 0x9d, 0xb1, 0x97, 0xaf, 0x2d, 0xac, 0x76, 0x81,
		0x1f, 0xfb, 0x4d, 0x53, 0xfe, 0x04, 0xe2, 0x48, 0xbe, 0xac,
		0x50, 0xe2, 0xb3, 0x74, 0x77, 0x5f, 0x48, 0xec, 0x26, 0xd0,
		0x9b, 0xb9, 0xa3, 0x28, 0x23, 0xa9, 0x2c, 0xc2, 0x0a, 0xb7,
		0xd3, 0x80, 0x87, 0x03, 0xa2, 0x3b, 0x74, 0x07, 0xaf, 0xa5,
		0x5f, 0x2b, 0x9b, 0x90, 0xa2, 0xf8, 0x89, 0x3c),
	.pub_r = chunk_from_chars(
		0x75, 0x3a, 0x06, 0x23, 0x48, 0x41, 0x6e, 0x90, 0x3b, 0x5b,
		0x3b, 0x25, 0x89, 0x38, 0xf1, 0xa4, 0x3f, 0xe0, 0x96, 0x2a,
		0xcb, 0x3c, 0xd2, 0x7a, 0x71, 0xb3, 0xed, 0x8a, 0xd4, 0xa5,
		0x62, 0x77, 0x4b, 0x6f, 0xf4, 0xf2, 0x29, 0x31, 0x2a, 0xfc,
		0xb4, 0x7b, 0x34, 0xfe, 0x9c, 0xb0, 0x83, 0x62, 0xe7, 0x45,
		0xc9, 0x93, 0x19, 0x89, 0xdb, 0x90, 0x99, 0xc5, 0x77, 0x85,
		0x06, 0x97, 0xa6, 0x2f, 0xde, 0x6d, 0x98, 0x01, 0xbc, 0x4f,
		0x51, 0x92, 0x94, 0x6f, 0x10, 0x3a, 0x7a, 0x56, 0x14, 0x48,
		0xad, 0x7d, 0x1d, 0x15, 0x0c, 0x8c, 0xda, 0xc9, 0x01, 0xf1,
		0x3b, 0xfd, 0x27, 0x09, 0x2e, 0xf7, 0xec, 0x0f, 0x82, 0x1e, /* 100 */
		0x0e, 0xa6, 0xb9, 0x1b, 0x63, 0x90, 0xc3, 0x3e, 0x7e, 0xf1,
		0xad, 0x5c, 0xaf, 0x6f, 0x6d, 0x9d, 0x3f, 0x25, 0x4f, 0xe9,
		0x53, 0xaf, 0x03, 0x6e, 0xdc, 0x24, 0xf3, 0x2c, 0x65, 0x67,
		0xc7, 0x08, 0x61, 0x80, 0x18, 0x7c, 0x19, 0x97, 0x44, 0x56,
		0x5e, 0xf0, 0xa2, 0x94, 0x7c, 0x59, 0x01, 0x94, 0x5b, 0x46,
		0xa8, 0x0b, 0x28, 0x6c, 0xa0, 0xfc, 0xa0, 0xad, 0xe4, 0x4a,
		0x2c, 0x87, 0x77, 0x7b, 0x44, 0x28, 0x25, 0xd4, 0xa2, 0x24,
		0x70, 0x69, 0x9a, 0x83, 0xf7, 0x65, 0xde, 0xe3, 0xeb, 0x03,
		0x14, 0x00, 0x4c, 0xba, 0x87, 0x87, 0xf2, 0x47, 0x4c, 0x3e,
		0x44, 0x67, 0x66, 0x85, 0x48, 0xb4, 0x12, 0xa6, 0x15, 0x22, /* 200 */
		0x0d, 0x39, 0x02, 0x07, 0x66, 0x59, 0x07, 0x3a, 0x64, 0x9e,
		0xba, 0x6e, 0xc4, 0xdc, 0x29, 0x07, 0x5b, 0x67, 0xff, 0x42,
		0xca, 0xe0, 0x1d, 0xcd, 0x39, 0x08, 0xff, 0x63, 0x03, 0xb1,
		0x76, 0xc4, 0xa3, 0xdc, 0x81, 0x33, 0xfb, 0x4c, 0x28, 0xa1,
		0xe4, 0x7e, 0xbe, 0x5f, 0x73, 0x24, 0x92, 0x7a, 0x40, 0x8d,
		0x75, 0xc5, 0x94, 0x13, 0x26, 0x91, 0xef, 0x9a, 0xee, 0x45,
		0xaa, 0xff, 0xfc, 0xae, 0x61, 0x34, 0xdb, 0x20, 0x96, 0x99,
		0xe9, 0x18, 0x30, 0x95, 0x37, 0x23, 0xaf, 0x3d, 0x2f, 0x3b,
		0x03, 0x69, 0x4b, 0xfa, 0x92, 0x92, 0x57, 0x8e, 0x66, 0xe2,
		0x89, 0xf0, 0x62, 0xe5, 0x2b, 0x2c, 0x23, 0xca, 0xcd, 0x8d, /* 300 */
		0xdd, 0x88, 0x92, 0xb4, 0xc9, 0x8e, 0x9d, 0x57, 0x62, 0x69,
		0x3e, 0xd1, 0xd7, 0xc8, 0x7e, 0x99, 0xac, 0x4e, 0x7c, 0x65,
		0xaf, 0xea, 0x99, 0xfa, 0x69, 0xd8, 0x46, 0xb2, 0xc7, 0x1f,
		0x76, 0xf1, 0x3e, 0x99, 0xb7, 0x23, 0x2c, 0x7c, 0x80, 0x8b,
		0x3a, 0x5e, 0x86, 0x2c, 0x50, 0x5a, 0x36, 0x48, 0x0a, 0x23,
		0x23, 0xdf, 0x69, 0x95, 0xa0, 0x1b, 0xfd, 0x1f, 0x4e, 0x06,
		0xc5, 0x0b, 0x17, 0x3c, 0x62, 0xce, 0x5a, 0x63, 0x82, 0xcd,
		0x14, 0x64, 0xb8, 0x60, 0x36, 0xb9, 0x74, 0x9c, 0xa4, 0xe1,
		0xa5, 0x0c, 0xc0, 0x77, 0x05, 0x41, 0x46, 0xac, 0x16, 0xdb,
		0xb5, 0x16, 0x71, 0x71, 0x6e, 0x62, 0x93, 0x17, 0xd6, 0xdc, /* 400 */
		0xbb, 0xbd, 0xb3, 0x01, 0x5f, 0x08, 0xa9, 0x71, 0x91, 0x97,
		0x92, 0xb1, 0x1d, 0xa8, 0x0a, 0xf9, 0xc3, 0xaa, 0x4c, 0xc2,
		0x63, 0x48, 0xd1, 0xae, 0x21, 0xbb, 0xf3, 0xb7, 0xda, 0x04,
		0x5e, 0x6e, 0x82, 0x89, 0x5d, 0xdc, 0xfb, 0xae, 0x76, 0xaf,
		0x50, 0x42, 0x71, 0x06, 0x8b, 0x0c, 0xfd, 0xb9, 0x0f, 0x00,
		0x24, 0x97, 0xe0, 0x0c, 0x9f, 0xf2, 0x95, 0x11, 0x63, 0x6f,
		0xcf, 0x59, 0xfb, 0xd2, 0xcc, 0x10, 0xec, 0xaa, 0xef, 0x89,
		0xff, 0x1b, 0x48, 0xc9, 0xce, 0x78, 0x22, 0x50, 0xf6, 0x31,
		0x47, 0x78, 0x38, 0x3b, 0xae, 0x32, 0xed, 0xf6, 0xaa, 0xa9,
		0x7a, 0x53, 0x71, 0xc6, 0xbd, 0x10, 0xcf, 0x17, 0xf4, 0x1b, /* 500 */
		0x1e, 0xb0, 0x90, 0x4d, 0xd1, 0xd2, 0xa2, 0x9b, 0x5c, 0x37,
		0xd3, 0x9c, 0x31, 0xb2, 0xb8, 0x5b, 0x8c, 0xa2, 0xde, 0x11,
		0xf7, 0x97, 0x03, 0xea, 0x45, 0x38, 0xc5, 0x5c, 0x22, 0x8e,
		0x3d, 0x60, 0x4a, 0xc7, 0x32, 0xaa, 0xee, 0x7a, 0x67, 0x9a,
		0xa5, 0x85, 0x1f, 0x64, 0xb1, 0x45, 0xe7, 0xe1, 0x69, 0x68,
		0x5c, 0x65, 0x1e, 0x0a, 0xf3, 0xf3, 0x11, 0x26, 0x98, 0x7b,
		0xf8, 0x27, 0x23, 0xad, 0xf4, 0x25, 0x6f, 0xab, 0x83, 0x48,
		0xc4, 0x5e, 0xba, 0xea, 0x73, 0x6a, 0x2b, 0x82, 0x66, 0x02,
		0xf5, 0x21, 0x5a, 0xbc, 0xf5, 0xbf, 0xf1, 0xa4, 0x72, 0x1c,
		0xd9, 0x9d, 0xb6, 0x46, 0xe9, 0xb5, 0x61, 0xbe, 0xe5, 0x59, /* 600 */
		0x8a, 0xf9, 0x8e, 0xfa, 0x79, 0x2e, 0xa6, 0x02, 0xad, 0x22,
		0xea, 0x06, 0x2c, 0x42, 0x66, 0xb9, 0x0c, 0x6d, 0x4b, 0x2b,
		0x8b, 0xd9, 0xa3, 0x8b, 0x60, 0xe2, 0x63, 0xe0, 0x44, 0x54,
		0x02, 0x2f, 0x75, 0xb7, 0x41, 0x81, 0x9c, 0xe7, 0xce, 0xc4,
		0x3e, 0x82, 0x05, 0x5e, 0x0e, 0x4c, 0x16, 0x0a, 0x59, 0xfa,
		0xb1, 0x13, 0x02, 0x87, 0xb8, 0xd8, 0xa7, 0xbc, 0x15, 0xb2,
		0x5a, 0xb5, 0xea, 0x50, 0x76, 0x76, 0x73, 0xa1, 0xf4, 0xc2,
		0x71, 0x88, 0x5d, 0x0c, 0x8c, 0xbe, 0x32, 0x3d, 0x60, 0x15,
		0xdb, 0xad, 0xde, 0x37, 0xf8, 0x8e, 0xb8, 0xd2, 0x24, 0xc3,
		0x3c, 0x97, 0xe7, 0x9c, 0xc6, 0xdc, 0xcd, 0xcd, 0x43, 0x93, /* 700 */
		0x06, 0xd8, 0x64, 0x9f, 0xca, 0x07, 0x15, 0x47, 0xca, 0x13,
		0x39, 0x8d, 0xd6, 0x75, 0xe0, 0x61, 0x7f, 0x7f, 0x15, 0x28,
		0x8f, 0xe8, 0x4d, 0x19, 0xb6, 0x41, 0x20, 0x93, 0x17, 0x03,
		0xaf, 0x1b, 0x16, 0x13, 0x07, 0xc7, 0x50, 0xfe, 0xeb, 0x97,
		0x7c, 0xe3, 0x72, 0x32, 0x9b, 0x87, 0xab, 0xab, 0x2d, 0x47,
		0xa0, 0x93, 0xc3, 0xc9, 0x17, 0x58, 0xc6, 0x2c, 0x8a, 0xa8,
		0x78, 0x6e, 0x6c, 0x30, 0x6c, 0xbf, 0x3f, 0x66),
	.shared = chunk_from_chars(
		0xfc, 0xe8, 0xe5, 0xeb, 0xf2, 0xb0, 0x07, 0xfc, 0x46, 0x60,
		0x17, 0xa8, 0xed, 0xf0, 0xf1, 0xa6, 0x9f, 0x5a, 0xf2, 0xea,
		0x49, 0x2a, 0x09, 0xd8, 0x08, 0xd2, 0x84, 0xc2, 0x2d, 0x9b,
		0x3d, 0x07, 0x2b, 0x9f, 0x89, 0x1b, 0x0c, 0x0f, 0x09, 0xe4,
		0x67, 0xd1, 0x6e, 0x33, 0x8d, 0x2e, 0x6d, 0xd1, 0xf5, 0x1b,
		0xdb, 0xc4, 0xe8, 0x64, 0x61, 0xf5, 0x49, 0x47, 0xd0, 0x07,
		0x0f, 0xbe, 0xc1, 0xfc, 0xe3, 0xe4, 0xf3, 0xd8, 0xa6, 0xeb,
		0x73, 0x8c, 0xb1, 0xd5, 0xcf, 0xc6, 0x54, 0x6d, 0xe2, 0x07,
		0xba, 0x55, 0x17, 0xe8, 0x2b, 0x39, 0xc2, 0xab, 0x30, 0x8d,
		0x2d, 0x48, 0xe1, 0xe7, 0x3f, 0x7f, 0x5c, 0xf8, 0x4a, 0xfa, /* 100 */
		0x7b, 0x71, 0xb3, 0xf2, 0x4c, 0x52, 0x45, 0x5d, 0xfd, 0xa0,
		0x94, 0x98, 0xe9, 0xde, 0x3f, 0x14, 0x52, 0x7f, 0xa5, 0xd6,
		0xf7, 0xa4, 0x67, 0x27, 0x4d, 0x84, 0x4e, 0xaf, 0x7f, 0x20,
		0xef, 0xc6, 0xaf, 0x5d, 0xac, 0xaf, 0xd7, 0xe1, 0x52, 0x02,
		0x6f, 0xd4, 0x84, 0x73, 0x15, 0x7b, 0x74, 0x89, 0x65, 0xcd,
		0xa7, 0xb8, 0xd5, 0x8d, 0x54, 0x94, 0x44, 0x88, 0xf6, 0xd2,
		0xa9, 0x7f, 0xa9, 0x12, 0xd1, 0x19, 0xa7, 0x75, 0x2f, 0xcf,
		0x3c, 0xb9, 0xc2, 0xc7, 0x66, 0x18, 0x91, 0x83, 0xd9, 0xa1,
		0x69, 0xe2, 0x09, 0xc0, 0x38, 0x7f, 0x7b, 0xff, 0x5e, 0x44,
		0x9d, 0x34, 0x3c, 0xef, 0xb3, 0xbc, 0x86, 0x57, 0xaa, 0x28, /* 200 */
		0x22, 0x09, 0x7a, 0xa1, 0x64, 0xad, 0xf0, 0x16, 0xcb, 0x15,
		0x8f, 0xaf, 0x9e, 0xb8, 0x0f, 0x9f, 0x53, 0x1c, 0x8a, 0xf9,
		0xf3, 0x2d, 0xee, 0xf7, 0x31, 0xa8, 0xcd, 0x44, 0x6a, 0x5f,
		0xea, 0x66, 0x95, 0x81, 0x96, 0x49, 0x83, 0xd7, 0x1d, 0x54,
		0xaf, 0x34, 0x81, 0x5d, 0x3c, 0x4e, 0x4e, 0xc5, 0xe6, 0x5c,
		0x04, 0x4d, 0x7a, 0x5c, 0x4b, 0x7b, 0x54, 0x14, 0xa1, 0x19,
		0x31, 0xe3, 0xda, 0x28, 0x4f, 0x8a, 0x3a, 0x99, 0x12, 0x3c,
		0x22, 0xb5, 0xce, 0xe7, 0x8b, 0x1c, 0xf0, 0x3a, 0xdf, 0x88,
		0x5b, 0xe5, 0xdb, 0x0a, 0x88, 0xcd, 0x84, 0x9d, 0x41, 0xdd,
		0xf1, 0x63, 0x00, 0xf3, 0x6c, 0x23, 0xa4, 0x4b, 0x81, 0x04, /* 300 */
		0x74, 0xd1, 0x35, 0x8f, 0xef, 0x99, 0x9d, 0xf9, 0x46, 0x7f,
		0x96, 0x98, 0x15, 0xbc, 0xa6, 0x25, 0x39, 0x6e, 0x32, 0xd1,
		0x7f, 0xa0, 0xcc, 0xe9, 0x60, 0x41, 0xc7, 0xcf, 0x0d, 0xd3,
		0x16, 0x99, 0xee, 0xb7, 0x63, 0xbb, 0xb2, 0x4c, 0x93, 0x7f,
		0x50, 0xd0, 0xfe, 0x90, 0x39, 0x33, 0x87, 0xc6, 0x17, 0x97,
		0x44, 0x83, 0xad, 0x3c, 0x20, 0x2f, 0xd5, 0xc0, 0x30, 0xf4,
		0xfa, 0xde, 0xdf, 0x42, 0x43, 0x16, 0x46, 0x6a, 0x6b, 0x36,
		0x59, 0xcd, 0xf1, 0x8c, 0x39, 0x82, 0xba, 0x54, 0xf6, 0x34,
		0x3f, 0x6b, 0xf2, 0xf2, 0x1f, 0x58, 0x58, 0xea, 0xdb, 0x8f,
		0x80, 0xa2, 0x4d, 0x01, 0xe8, 0x2d, 0x70, 0x0c, 0x58, 0xf9, /* 400 */
		0x56, 0x4f, 0x6e, 0xea, 0xe2, 0x98, 0x0c, 0xc7, 0x01, 0x74,
		0x40, 0x25, 0x24, 0xcc, 0x87, 0x16, 0x98, 0x67, 0x5a, 0x7e,
		0x95, 0x87, 0xa1, 0x41, 0x0e, 0xa6, 0x22, 0x89, 0x13, 0x26,
		0x0f, 0x6f, 0x81, 0xdc, 0xd5, 0x3d, 0x75, 0xe0, 0xd6, 0x71,
		0xe7, 0xde, 0x0a, 0x6c, 0x62, 0x3b, 0x64, 0xab, 0x2f, 0x67,
		0xa0, 0xad, 0xdc, 0x32, 0x70, 0x1c, 0xa1, 0xfa, 0xe9, 0x08,
		0xed, 0x5d, 0x90, 0x09, 0xad, 0x42, 0x74, 0x5d, 0xcd, 0x12,
		0x24, 0xfb, 0x14, 0xe8, 0xb9, 0x89, 0xa4, 0x4f, 0x0b, 0x07,
		0xa9, 0x4a, 0x7a, 0x27, 0xce, 0x19, 0x8d, 0x32, 0x6f, 0x74,
		0x9a, 0xc5, 0x10, 0x91, 0xf1, 0xfc, 0x03, 0xfa, 0x1c, 0x87, /* 500 */
		0xe8, 0x70, 0x5c, 0x36, 0xa8, 0xef, 0x2c, 0xc7, 0xb6, 0xe4,
		0x47, 0xac, 0xb7, 0x3d, 0x0e, 0x44, 0x12, 0xbd, 0xa2, 0xf9,
		0xdb, 0x49, 0x5f, 0xc5, 0xe6, 0xdf, 0x7a, 0x37, 0x87, 0x32,
		0x73, 0xb3, 0x6d, 0x08, 0xac, 0xef, 0xb8, 0xa3, 0x0a, 0x14,
		0xb6, 0xd7, 0x7f, 0x76, 0x0f, 0x02, 0x9b, 0x93, 0x45, 0x7b,
		0xee, 0xd4, 0x92, 0x53, 0x40, 0x6c, 0x9e, 0xe1, 0x52, 0xe4,
		0x22, 0x31, 0xcf, 0xcb, 0xc5, 0x30, 0x11, 0xe0, 0xb6, 0xe4,
		0x17, 0xad, 0x03, 0xe9, 0x8d, 0xaf, 0x2e, 0xc8, 0x5a, 0x2c,
		0x89, 0x49, 0x9d, 0xde, 0x67, 0xc4, 0x03, 0x1d, 0x91, 0x8f,
		0x30, 0x7d, 0x11, 0xd1, 0x93, 0x8c, 0x7b, 0xb9, 0x17, 0x71, /* 600 */
		0x7c, 0x3b, 0x14, 0x34, 0x03, 0xba, 0x76, 0x8a, 0x4b, 0xe0,
		0xd0, 0x93, 0xc9, 0x25, 0x59, 0xce, 0x3f, 0xcc, 0x5d, 0x38,
		0xfe, 0x18, 0xfa, 0xbb, 0x0d, 0xde, 0xe0, 0xd4, 0x17, 0x3e,
		0x18, 0x27, 0x58, 0x73, 0x86, 0x89, 0x71, 0xa0, 0x7f, 0xcd,
		0xd4, 0x8e, 0x54, 0xcc, 0x3f, 0x63, 0x1e, 0x44, 0x3d, 0xb9,
		0x39, 0x77, 0x4c, 0xb8, 0xaa, 0x57, 0x5f, 0x7b, 0xd2, 0xa9,
		0x86, 0x9c, 0x15, 0xbe, 0xa7, 0xf1, 0xea, 0xa9, 0xdf, 0x95,
		0x32, 0x2c, 0xbf, 0x93, 0xb0, 0x31, 0xaf, 0x6e, 0x74, 0xe0,
		0x37, 0x7c, 0x94, 0xf0, 0x4f, 0x6c, 0x44, 0x3c, 0xd9, 0x1c,
		0x3b, 0x48, 0x24, 0x8b, 0x28, 0x2f, 0x65, 0x54, 0xce, 0x69, /* 700 */
		0xf4, 0x5b, 0xb6, 0x11, 0xef, 0xab, 0xec, 0x45, 0x4d, 0x10,
		0x58, 0xa7, 0xf2, 0xa7, 0xc7, 0x1a, 0x9f, 0xfa, 0xdc, 0xcd,
		0x54, 0xfa, 0x0d, 0xe8, 0x96, 0xbc, 0x0d, 0x55, 0x0a, 0x49,
		0xf1, 0x2a, 0x31, 0x28, 0x15, 0x26, 0x56, 0xf2, 0xa0, 0x6c,
		0x84, 0x51, 0xbd, 0x6b, 0xee, 0x20, 0xc5, 0xd6, 0x4b, 0x36,
		0x5f, 0x7b, 0xb8, 0xd0, 0xeb, 0x41, 0xb6, 0x7c, 0xfb, 0x9d,
		0x9d, 0xfd, 0x62, 0x0e, 0xb7, 0x99, 0xca, 0x17)
};

ke_test_vector_t modp8192 = {
	.method = MODP_8192_BIT,
	.seed = chunk_from_chars(
		0xa6, 0x48, 0x81, 0x47, 0x45, 0xfe, 0x1e, 0xd1, 0x98, 0x9b,
		0x75, 0xba, 0x6d, 0xd5, 0x01, 0xe4, 0x4e, 0x77, 0x6d, 0xc9,
		0x97, 0xa0, 0xae, 0x27, 0x37, 0x64, 0x61, 0xb0, 0xee, 0x79,
		0x65, 0x94, 0xc2, 0xe6, 0xdb, 0x07, 0xe5, 0xf9, 0xd8, 0x7d,
		0x94, 0x4d, 0x37, 0x01, 0x22, 0x38, 0xe5, 0x70, 0xaf, 0x52,
		0x3a, 0x47, 0xf3, 0xe8, 0x6e, 0x2a, 0x4c, 0xd0, 0xdb, 0x3a,
		0x49, 0x40, 0xcf, 0x0b, /* - */
		0xd8, 0xf2, 0xb0, 0x9f, 0x17, 0xbe, 0x6f, 0x13, 0x64, 0x5c,
		0xb4, 0x57, 0xbe, 0x0a, 0xa8, 0x02, 0x02, 0xe8, 0x43, 0xf5,
		0x10, 0xd2, 0x93, 0xf9, 0x32, 0x55, 0x98, 0x2f, 0x89, 0x87,
		0x26, 0x61, 0x79, 0xd6, 0x01, 0x90, 0x95, 0x70, 0xa3, 0xf9,
		0xcc, 0x58, 0x50, 0x8b, 0x62, 0xf8, 0x8d, 0x64, 0xe0, 0xd0,
		0x59, 0xf1, 0xa2, 0xed, 0xea, 0xd4, 0x42, 0x82, 0x04, 0xcc,
		0x09, 0x2a, 0x2d, 0x82),
	.pub_i = chunk_from_chars(
		0x1f, 0x6d, 0x66, 0x0a, 0xfb, 0x04, 0x87, 0x42, 0x55, 0xe6,
		0x83, 0xee, 0x89, 0x9a, 0xd2, 0xfd, 0xbc, 0xf7, 0x35, 0x7f,
		0xdf, 0xe2, 0x47, 0xef, 0x3b, 0x2f, 0xf9, 0x41, 0x19, 0x48,
		0x45, 0x26, 0x44, 0x53, 0x41, 0xc5, 0xdf, 0xea, 0x28, 0xa7,
		0x45, 0xa0, 0xff, 0x56, 0x95, 0x13, 0x24, 0x72, 0x34, 0x1b,
		0x11, 0xb2, 0xe6, 0x9b, 0xd8, 0xe9, 0x98, 0x18, 0xf0, 0x5c,
		0x22, 0x74, 0x26, 0x65, 0x9c, 0xf0, 0x34, 0x81, 0x55, 0x50,
		0x1d, 0xce, 0x5a, 0x7b, 0x5d, 0x11, 0xae, 0xaf, 0xfb, 0xd0,
		0x52, 0xb5, 0xff, 0x8a, 0xc8, 0xe3, 0x5e, 0x78, 0x50, 0x29,
		0x79, 0x53, 0x9e, 0xf4, 0xe4, 0xa7, 0xa9, 0x51, 0x55, 0x1a, /* 100 */
		0xcc, 0x10, 0x02, 0x72, 0x6b, 0xa8, 0xee, 0x95, 0x95, 0x70,
		0x19, 0x92, 0x34, 0x15, 0xd8, 0xec, 0x61, 0x27, 0xbe, 0x25,
		0x2c, 0x44, 0x65, 0xb0, 0xef, 0xae, 0x8a, 0xee, 0xcd, 0x39,
		0x0f, 0x33, 0x55, 0xcf, 0x94, 0x52, 0x5e, 0xef, 0x38, 0xdb,
		0x92, 0x62, 0x79, 0xb8, 0xfd, 0xce, 0xe1, 0x3c, 0x43, 0x8e,
		0xdd, 0xb2, 0xa5, 0x22, 0x44, 0xa7, 0x1b, 0xda, 0x1e, 0x85,
		0x30, 0x91, 0x82, 0xfa, 0x8c, 0x57, 0xff, 0x0c, 0xd4, 0x06,
		0x9d, 0x4f, 0xfb, 0xcc, 0x42, 0xe5, 0xe6, 0x60, 0xc1, 0xf0,
		0x07, 0x3c, 0x85, 0x6a, 0xc6, 0x11, 0xcb, 0xf7, 0xf9, 0xf3,
		0x2c, 0x95, 0x67, 0xa1, 0x8e, 0x44, 0x94, 0xed, 0xe7, 0x42, /* 200 */
		0x7b, 0x1a, 0x18, 0x16, 0x55, 0xdb, 0xc7, 0x91, 0xc7, 0x9d,
		0x54, 0x7d, 0x88, 0xfd, 0x49, 0x45, 0x9b, 0xd3, 0x56, 0xee,
		0x47, 0x26, 0xa9, 0x5c, 0xd3, 0x7b, 0xf2, 0x42, 0xe4, 0x8a,
		0x94, 0xd9, 0x22, 0xcc, 0x06, 0x66, 0xce, 0x4f, 0x29, 0x91,
		0x2a, 0x1e, 0x4e, 0xc8, 0x8e, 0x3e, 0x40, 0xde, 0xdc, 0x95,
		0xe1, 0x2c, 0xe2, 0x50, 0x47, 0x2e, 0x58, 0x37, 0x49, 0x32,
		0xe4, 0x64, 0x81, 0xe7, 0x03, 0x11, 0x81, 0x13, 0xb7, 0xdd,
		0x6f, 0xef, 0xae, 0xf5, 0xc5, 0xe6, 0x1c, 0x13, 0x6b, 0x76,
		0xa1, 0xb0, 0xbf, 0x4d, 0xf0, 0x01, 0xc3, 0x09, 0xd8, 0x59,
		0xff, 0x13, 0x34, 0x6d, 0x52, 0x11, 0x9b, 0xa6, 0xc9, 0xca, /* 300 */
		0x38, 0xe6, 0x71, 0x80, 0x39, 0xae, 0x37, 0x4b, 0x1c, 0xe1,
		0x13, 0xfb, 0xc6, 0x91, 0x8f, 0x32, 0xc3, 0x6a, 0x04, 0xd8,
		0xc6, 0x80, 0x08, 0x4e, 0xef, 0xf6, 0x80, 0xf9, 0x8f, 0xda,
		0xaf, 0x27, 0x79, 0x9b, 0x21, 0x2d, 0x2d, 0xea, 0x87, 0x4b,
		0x4d, 0xeb, 0x5a, 0x87, 0xfb, 0x72, 0x36, 0xe2, 0x3e, 0x9b,
		0x42, 0xcf, 0xa9, 0xeb, 0x49, 0x41, 0xe6, 0xc2, 0xb8, 0xc0,
		0xad, 0xbd, 0xb3, 0x61, 0xc1, 0x61, 0x5f, 0x9c, 0xb6, 0xbd,
		0x8f, 0x99, 0x3a, 0xe8, 0xca, 0x86, 0xb8, 0xd8, 0x7f, 0x2f,
		0xb2, 0x33, 0xc0, 0x9f, 0xd3, 0x8e, 0x44, 0xdc, 0x6d, 0x0c,
		0x19, 0x89, 0xb9, 0x05, 0xc6, 0x36, 0xf4, 0x7a, 0xc3, 0x06, /* 400 */
		0x46, 0xa4, 0x22, 0x6e, 0xef, 0x76, 0xba, 0x92, 0xfd, 0xaf,
		0xce, 0x71, 0xbc, 0x0c, 0x05, 0xfe, 0xec, 0x8f, 0x8d, 0xfb,
		0x59, 0x46, 0xf9, 0x7c, 0xc1, 0x1f, 0x1c, 0x52, 0x62, 0x4d,
		0x01, 0x57, 0x4e, 0x86, 0x03, 0x94, 0x8b, 0xba, 0xf7, 0xbd,
		0xca, 0xbb, 0x15, 0xaa, 0x61, 0xdd, 0xbc, 0x32, 0xdf, 0x9e,
		0x5f, 0xad, 0x3b, 0xf8, 0xf6, 0xfb, 0x4d, 0x1e, 0x9c, 0x86,
		0x9e, 0x7e, 0x0f, 0xaf, 0xf6, 0x91, 0x7e, 0x08, 0xf4, 0xfb,
		0x55, 0xd8, 0xe0, 0x4b, 0xd9, 0xcd, 0x23, 0x57, 0x83, 0x58,
		0x59, 0xd7, 0x56, 0x93, 0xee, 0x14, 0x2f, 0x2b, 0xd2, 0x83,
		0xa4, 0xce, 0x45, 0xa2, 0x90, 0xba, 0x15, 0x4d, 0xca, 0x0c, /* 500 */
		0x8b, 0x29, 0x7a, 0xe5, 0xbc, 0xba, 0xa0, 0x45, 0xd0, 0x08,
		0x27, 0x75, 0x61, 0x3a, 0x83, 0x99, 0x2f, 0x39, 0xc6, 0x41,
		0x03, 0xc0, 0xb7, 0xd6, 0xfb, 0x5b, 0x94, 0x64, 0xc3, 0xfe,
		0xca, 0x32, 0x1c, 0xe3, 0x66, 0xc8, 0xb4, 0x49, 0x44, 0x1c,
		0x63, 0xb0, 0xaa, 0x18, 0x31, 0x4b, 0x15, 0x8a, 0xda, 0x77,
		0xc5, 0xfd, 0xea, 0x33, 0x6c, 0x9a, 0x45, 0x4c, 0x24, 0xb5,
		0x1c, 0xd2, 0x5d, 0x56, 0x98, 0x3c, 0xdf, 0xb1, 0x5f, 0x10,
		0xee, 0xc1, 0x17, 0xec, 0xbe, 0x7c, 0xdb, 0x24, 0xdd, 0xdb,
		0x22, 0xf6, 0x47, 0x8a, 0xfc, 0xc5, 0xb5, 0xa8, 0x9a, 0x8f,
		0xb8, 0x27, 0x72, 0xa1, 0xd4, 0x5e, 0x43, 0xcd, 0x7b, 0x33, /* 600 */
		0x2e, 0xe3, 0x09, 0x94, 0x77, 0x20, 0xfe, 0xba, 0xae, 0x2e,
		0x9b, 0xc1, 0x43, 0xdd, 0x9d, 0x44, 0xd8, 0xd6, 0xfe, 0xff,
		0xe3, 0x0f, 0xf6, 0xd3, 0x71, 0xa1, 0xf8, 0xda, 0x1c, 0xff,
		0x41, 0x21, 0x6f, 0x07, 0xc9, 0x55, 0x99, 0x6f, 0x0a, 0xef,
		0xd6, 0x5a, 0x6c, 0xa5, 0xdd, 0xba, 0x98, 0x46, 0x30, 0x43,
		0xad, 0x27, 0xe4, 0xf1, 0x1e, 0x3a, 0x89, 0x4a, 0xb3, 0xc8,
		0x6d, 0xf7, 0xe9, 0x2f, 0x48, 0xd0, 0xd7, 0x29, 0x38, 0x5c,
		0xe7, 0xac, 0xbc, 0x3f, 0x65, 0x5e, 0x23, 0xdd, 0xc1, 0xad,
		0x73, 0xed, 0x1a, 0xee, 0x81, 0xf3, 0x63, 0x29, 0x7e, 0x72,
		0x8f, 0x1a, 0xfc, 0x2d, 0x03, 0xf9, 0xbb, 0x3c, 0x38, 0x42, /* 700 */
		0xc2, 0xfb, 0x53, 0x2f, 0x56, 0xd6, 0xca, 0xb9, 0xeb, 0xa4,
		0x17, 0x46, 0xdb, 0x53, 0x56, 0xf0, 0xdd, 0x1d, 0x8a, 0xfc,
		0x03, 0x06, 0x4d, 0x8c, 0x97, 0x7e, 0xf0, 0xc6, 0x5d, 0x6d,
		0x5a, 0x23, 0xed, 0xee, 0xf9, 0x11, 0xed, 0x04, 0x34, 0x0c,
		0x04, 0xa0, 0x60, 0xf9, 0xa8, 0xfe, 0x8f, 0xfa, 0xd6, 0xf3,
		0x27, 0x3d, 0x32, 0x48, 0xbe, 0x3b, 0x56, 0x3a, 0xe8, 0x76,
		0xe9, 0x54, 0xe7, 0x81, 0xef, 0xe3, 0x8f, 0xd9, 0x03, 0x42,
		0x5b, 0xa7, 0xd2, 0x69, 0x96, 0x39, 0x05, 0x8f, 0x41, 0x25,
		0x35, 0x3a, 0x56, 0x66, 0x5a, 0xc8, 0x36, 0xda, 0x84, 0xc5,
		0xe9, 0x2e, 0x55, 0xac, 0xe9, 0xeb, 0xdc, 0x8f, 0xd8, 0x26, /* 800 */
		0x06, 0x3b, 0x61, 0x7f, 0xd7, 0x78, 0x4e, 0x5b, 0xe5, 0xd7,
		0xca, 0x76, 0xf0, 0xd0, 0x71, 0xf6, 0x45, 0x21, 0x30, 0xdf,
		0xfc, 0x69, 0x82, 0xcf, 0xc3, 0x02, 0xda, 0x4f, 0x4e, 0x51,
		0x59, 0x4f, 0x56, 0xd1, 0x60, 0x5a, 0xd9, 0x12, 0x21, 0x7a,
		0xa9, 0x99, 0x81, 0x5a, 0xa8, 0xa8, 0x10, 0x94, 0xd7, 0x3e,
		0x58, 0x7e, 0xd9, 0xde, 0xbc, 0xf5, 0x83, 0xf0, 0x92, 0x4f,
		0xe6, 0xdd, 0x17, 0xb3, 0x9c, 0xdd, 0xd5, 0x7b, 0xc0, 0xb4,
		0xb5, 0x8c, 0xae, 0x7b, 0x63, 0x1a, 0xaa, 0xd0, 0x93, 0x55,
		0x31, 0x28, 0xd7, 0x8e, 0xec, 0x8c, 0x28, 0x64, 0x81, 0x9e,
		0x2e, 0x47, 0xa9, 0xb3, 0xa0, 0x06, 0x2a, 0x3d, 0xfe, 0x27, /* 900 */
		0x40, 0x9c, 0x13, 0x96, 0xa0, 0xff, 0x65, 0xd9, 0x05, 0xd8,
		0x92, 0xa2, 0x63, 0x0e, 0xbf, 0xe2, 0xc5, 0xe3, 0x51, 0xbf,
		0x20, 0x74, 0xdc, 0xbe, 0x5a, 0xfb, 0xd5, 0x5c, 0x1b, 0x8f,
		0x41, 0x21, 0x43, 0xaf, 0xdf, 0x35, 0xd1, 0x46, 0x80, 0x16,
		0xf0, 0x28, 0x5d, 0x55, 0x52, 0x0c, 0x05, 0x24, 0x63, 0x68,
		0x5d, 0x50, 0x7a, 0x50, 0xde, 0x2d, 0xfa, 0xae, 0xd0, 0x71,
		0x9a, 0xad, 0x2a, 0x56, 0xb6, 0xa2, 0x52, 0xbc, 0xe1, 0x93,
		0xc4, 0x93, 0x36, 0xc6, 0x08, 0xec, 0xf3, 0x26, 0x14, 0x0b,
		0xa5, 0x4c, 0x9e, 0x15, 0xf3, 0x48, 0xf6, 0xd1, 0xd4, 0x45,
		0x41, 0xd7, 0xb7, 0x5c, 0xd9, 0x89, 0x89, 0xe6, 0x62, 0xe6, /* 1000 */
		0x4b, 0x9f, 0x3a, 0xf7, 0xc7, 0x77, 0xd7, 0xc3, 0xae, 0x18,
		0x3c, 0x5c, 0xee, 0x2a, 0xf1, 0x80, 0x8a, 0x73, 0x83, 0xa4,
		0x0e, 0x63, 0x12, 0x04),
	.pub_r = chunk_from_chars(
		0x8e, 0x2b, 0x1b, 0xac, 0x90, 0x76, 0x05, 0x3f, 0xf4, 0xd5,
		0xde, 0xa3, 0xad, 0xf0, 0xe9, 0x40, 0x27, 0x69, 0xb2, 0x65,
		0x1a, 0x54, 0x5c, 0x89, 0x0a, 0x82, 0xbd, 0x59, 0xc3, 0x4f,
		0x55, 0x02, 0x77, 0xdd, 0xac, 0x4c, 0x3b, 0x43, 0xb5, 0x34,
		0xf5, 0xa2, 0x15, 0x85, 0xba, 0xa2, 0x40, 0x6f, 0xb7, 0x37,
		0x6c, 0xdb, 0x3d, 0xd1, 0xc0, 0xc9, 0xa0, 0x6c, 0x17, 0x3c,
		0xa3, 0xc1, 0xcf, 0x7f, 0x86, 0x86, 0xcb, 0xaf, 0x9a, 0x5c,
		0x27, 0x36, 0x8e, 0x47, 0xac, 0x3f, 0x30, 0x46, 0xe7, 0xd9,
		0xd9, 0x56, 0x54, 0x22, 0xfe, 0x43, 0xb6, 0xc9, 0x04, 0xf9,
		0xd0, 0x63, 0x02, 0x02, 0xcd, 0x7d, 0x77, 0xcc, 0x21, 0x1b, /* 100 */
		0x7b, 0xea, 0x4a, 0xc1, 0x9c, 0xdb, 0x68, 0x0e, 0x34, 0x00,
		0x43, 0x4b, 0xd1, 0x66, 0x03, 0x07, 0x42, 0xc9, 0x79, 0x7f,
		0x81, 0x0f, 0xcc, 0xff, 0x4b, 0x2c, 0x00, 0xf9, 0xb9, 0xd4,
		0x70, 0xde, 0x92, 0x65, 0x45, 0x32, 0xac, 0x64, 0x76, 0x49,
		0xe6, 0xc0, 0x26, 0x77, 0xbc, 0xb8, 0x09, 0xa6, 0xbd, 0xa1,
		0x22, 0x73, 0x13, 0x4b, 0x1c, 0x7c, 0xa8, 0x88, 0xe1, 0x69,
		0xb1, 0xe2, 0x64, 0x48, 0x3f, 0x4b, 0x2f, 0xd8, 0xa3, 0xb4,
		0xf6, 0x4e, 0x5a, 0x79, 0x06, 0xf0, 0xcb, 0x72, 0x9c, 0x72,
		0x0e, 0xe6, 0x39, 0xa6, 0x69, 0x7b, 0xa3, 0x32, 0x9e, 0xa0,
		0x81, 0xb0, 0xd6, 0xa2, 0x10, 0xee, 0xbb, 0x5b, 0xd5, 0x51, /* 200 */
		0xd0, 0xd6, 0xbf, 0x28, 0x80, 0x0c, 0x67, 0xf4, 0x38, 0xf4,
		0xc3, 0x16, 0x3b, 0x83, 0xd2, 0x6c, 0xd3, 0xf3, 0x02, 0x34,
		0x64, 0xeb, 0xa1, 0x6c, 0xb3, 0xa5, 0x13, 0x6a, 0x64, 0xb5,
		0xa6, 0x3a, 0x1b, 0x63, 0x5e, 0xe7, 0x03, 0x96, 0xdb, 0x37,
		0x4f, 0xc8, 0xb8, 0x60, 0x86, 0x45, 0x30, 0x61, 0x97, 0xfa,
		0x8e, 0x3d, 0xae, 0x48, 0xa4, 0x7d, 0xfe, 0x72, 0x6a, 0xe9,
		0x98, 0xeb, 0x77, 0x13, 0x4a, 0x4e, 0x6a, 0xae, 0x24, 0xf2,
		0xd2, 0xad, 0xa6, 0xca, 0x7e, 0x98, 0x38, 0x53, 0x24, 0xea,
		0x98, 0x09, 0x2c, 0x5b, 0x0f, 0x46, 0x9c, 0x6a, 0x50, 0x0c,
		0x46, 0x27, 0x98, 0x51, 0x56, 0x06, 0x6e, 0xca, 0xe9, 0xb8, /* 300 */
		0x47, 0xe7, 0x20, 0xd7, 0x71, 0x38, 0xdc, 0x17, 0x74, 0x4e,
		0x0b, 0x9d, 0xa6, 0xe3, 0x8e, 0x69, 0x28, 0x2a, 0x90, 0x45,
		0x3e, 0x72, 0xdf, 0x62, 0x88, 0x3a, 0x8a, 0x04, 0x59, 0x05,
		0x43, 0x2a, 0xa3, 0x22, 0x90, 0x1f, 0x62, 0xa2, 0x46, 0x03,
		0x90, 0x81, 0xd8, 0x1a, 0x12, 0x77, 0x37, 0x08, 0x34, 0x58,
		0x0d, 0x0b, 0x5d, 0x02, 0x5f, 0xa3, 0x66, 0xfb, 0x6e, 0xeb,
		0x22, 0x5c, 0xe3, 0xbc, 0xcc, 0x8d, 0xa5, 0x94, 0xe8, 0x14,
		0xd5, 0x08, 0xfe, 0x8b, 0x75, 0x6a, 0xf1, 0x9a, 0xde, 0x32,
		0x26, 0x10, 0xb4, 0xb9, 0x9f, 0x5d, 0x60, 0x83, 0x66, 0xdf,
		0xfc, 0xfe, 0x4d, 0xdc, 0xd9, 0x95, 0xcf, 0xa9, 0x89, 0xc4, /* 400 */
		0xe6, 0x92, 0x8a, 0xd5, 0x99, 0x35, 0x7e, 0xc5, 0x5b, 0x42,
		0x62, 0x92, 0x3f, 0x7e, 0x2a, 0x32, 0xd0, 0x64, 0x8b, 0x0a,
		0x61, 0x0f, 0x3d, 0xa8, 0x83, 0xb8, 0x95, 0x77, 0x34, 0x3c,
		0xd0, 0x31, 0xe6, 0xf9, 0x01, 0x69, 0x2f, 0x3f, 0xb4, 0xa1,
		0x03, 0x18, 0x1b, 0xf1, 0xbd, 0xac, 0xe1, 0x8d, 0x61, 0xb8,
		0xd0, 0xf6, 0x7f, 0xd9, 0x7a, 0x3f, 0x26, 0xc1, 0x81, 0x16,
		0x80, 0x93, 0xe6, 0x65, 0x56, 0x5b, 0xc1, 0x56, 0xfb, 0xf1,
		0x65, 0xd9, 0x5a, 0x21, 0xe9, 0xf2, 0xd2, 0xde, 0x7d, 0x89,
		0x12, 0xa2, 0x4f, 0x8f, 0xd9, 0x61, 0x0c, 0x51, 0xbf, 0xfc,
		0x36, 0x65, 0x36, 0x5b, 0x51, 0x20, 0x90, 0x42, 0x3a, 0x99, /* 500 */
		0xd1, 0x88, 0x21, 0xf1, 0xd7, 0x1a, 0xe1, 0xed, 0xca, 0xdd,
		0x08, 0x4e, 0xb5, 0x29, 0xa3, 0x72, 0x83, 0xed, 0x85, 0x15,
		0xe5, 0x20, 0x5c, 0x6e, 0xbb, 0x39, 0x2a, 0x38, 0x12, 0xde,
		0xe0, 0x5a, 0x1c, 0x10, 0x7b, 0xb2, 0x5b, 0x46, 0xa2, 0x80,
		0xb4, 0x3e, 0x59, 0x1a, 0x7d, 0x09, 0x70, 0x9d, 0xbc, 0x6f,
		0x76, 0x8a, 0xf4, 0xb4, 0xc9, 0x35, 0x26, 0xfb, 0xcc, 0x7a,
		0xb5, 0x52, 0x8b, 0x06, 0xe3, 0x6f, 0x50, 0xb1, 0xf6, 0x4d,
		0xf4, 0xbd, 0xa9, 0x1e, 0x39, 0x37, 0x56, 0x1b, 0x25, 0x62,
		0x0f, 0x7a, 0x4a, 0xfb, 0xe1, 0xff, 0xf5, 0x28, 0xf8, 0x40,
		0xd4, 0xf2, 0x99, 0xd1, 0xad, 0xd9, 0x23, 0x5b, 0x4e, 0x3b, /* 600 */
		0x8a, 0x57, 0x87, 0x54, 0x23, 0xd7, 0x73, 0x54, 0xf9, 0xa0,
		0x0e, 0xa5, 0x49, 0x5f, 0x8e, 0x93, 0x8e, 0x41, 0xeb, 0x69,
		0x7b, 0xef, 0x48, 0xeb, 0x1b, 0xb6, 0x6b, 0x48, 0x29, 0xff,
		0x4e, 0x89, 0xb1, 0xf2, 0xbd, 0xef, 0x77, 0x7a, 0x92, 0x71,
		0x97, 0xac, 0x21, 0x76, 0x6e, 0x54, 0xd3, 0x89, 0xe2, 0xca,
		0x07, 0xbe, 0x00, 0xe8, 0xb4, 0xd4, 0xd5, 0x63, 0xf1, 0x4b,
		0xbb, 0x76, 0x16, 0xc8, 0xdb, 0xf1, 0x14, 0x18, 0x4b, 0xbe,
		0x39, 0x00, 0x75, 0xf2, 0x85, 0x2b, 0x8e, 0xd8, 0xc0, 0x81,
		0x12, 0x9b, 0xe4, 0x9f, 0x98, 0x74, 0x6f, 0x70, 0xa8, 0xa4,
		0x7f, 0x14, 0x66, 0x21, 0x91, 0x61, 0x86, 0x2b, 0xa6, 0x01, /* 700 */
		0xf5, 0xd1, 0x00, 0x6c, 0x7b, 0x3a, 0x39, 0xb8, 0x3b, 0x6c,
		0xe8, 0x4b, 0x16, 0xd0, 0x99, 0xcc, 0x2a, 0x9f, 0x63, 0xdd,
		0x77, 0xc1, 0x1d, 0x14, 0x98, 0x38, 0xca, 0x3f, 0x5a, 0xdf,
		0xc4, 0x9d, 0xf5, 0xe6, 0xa8, 0x2a, 0x4b, 0xfb, 0x53, 0x68,
		0x6e, 0x31, 0x39, 0x12, 0x07, 0x2b, 0x7e, 0xe3, 0x78, 0x3d,
		0x23, 0x79, 0xc4, 0x2b, 0xc1, 0xac, 0x29, 0x20, 0x59, 0x91,
		0xf0, 0xd8, 0xfb, 0x9d, 0x20, 0x5b, 0x21, 0x70, 0x0e, 0xa8,
		0x46, 0xca, 0xfc, 0x09, 0x38, 0x12, 0x1c, 0x2c, 0xa4, 0x67,
		0xa8, 0x94, 0x04, 0x87, 0xa4, 0xaa, 0x2c, 0x25, 0xea, 0x57,
		0x35, 0xc2, 0x4e, 0xa9, 0x15, 0x0e, 0x7c, 0x46, 0xe3, 0xdf, /* 800 */
		0x40, 0x77, 0x81, 0xd6, 0xf3, 0x9c, 0xee, 0xd4, 0x22, 0x3b,
		0xe9, 0x20, 0x98, 0x1a, 0x18, 0x4e, 0xd3, 0x3d, 0xe6, 0xd0,
		0xfc, 0xcc, 0x2d, 0x5c, 0x72, 0x69, 0xe7, 0xc4, 0xca, 0x12,
		0xe9, 0xb7, 0xbe, 0x84, 0x31, 0x9e, 0x5d, 0x6f, 0xcf, 0x91,
		0x37, 0x15, 0xeb, 0xf8, 0x08, 0xba, 0x50, 0xc3, 0xb4, 0xad,
		0x36, 0x09, 0xe0, 0xd3, 0x61, 0xea, 0x9b, 0xe6, 0x28, 0x56,
		0x33, 0x73, 0x2b, 0xe7, 0x51, 0x52, 0xc5, 0x40, 0x66, 0x34,
		0x47, 0x03, 0xe4, 0xd9, 0xff, 0x4b, 0x23, 0xaf, 0x80, 0x63,
		0xbd, 0x30, 0xe0, 0x2d, 0xe2, 0x08, 0x4a, 0x09, 0xd3, 0x04,
		0x0a, 0x4d, 0xbc, 0x7b, 0xac, 0x80, 0x94, 0x8c, 0x4d, 0xd7, /* 900 */
		0x4a, 0x35, 0x72, 0xa4, 0x18, 0x66, 0xf2, 0x0e, 0x01, 0x6a,
		0x0f, 0x31, 0x17, 0x31, 0x07, 0xb9, 0x65, 0x68, 0xd6, 0xac,
		0x2f, 0x53, 0x36, 0x01, 0x3f, 0x06, 0x18, 0x77, 0xae, 0xbe,
		0xb4, 0xac, 0x60, 0x9a, 0x73, 0x3d, 0x7d, 0x40, 0xdc, 0xed,
		0x19, 0x27, 0x03, 0x57, 0x0a, 0xca, 0xe4, 0xcc, 0xc6, 0xfe,
		0x3b, 0x7a, 0x9d, 0x73, 0xb2, 0xa4, 0xf1, 0x21, 0x32, 0x11,
		0x21, 0x86, 0x77, 0x1e, 0x25, 0x96, 0xc8, 0xb0, 0xce, 0xd5,
		0x1e, 0x4c, 0xea, 0xd0, 0x04, 0xde, 0x67, 0xa8, 0x36, 0x23,
		0x89, 0x85, 0x7b, 0xa9, 0x2a, 0x0c, 0x52, 0xd5, 0x63, 0xbe,
		0x17, 0xdb, 0x82, 0xd5, 0xf6, 0x51, 0xc3, 0x9a, 0x9f, 0xab, /* 1000 */
		0x58, 0x46, 0x87, 0x0c, 0xb3, 0xc1, 0x23, 0x87, 0x61, 0xac,
		0x00, 0xed, 0x46, 0xe5, 0x79, 0x2c, 0x5b, 0xd6, 0x6e, 0xaa,
		0xee, 0x4b, 0x08, 0xb2),
	.shared = chunk_from_chars(
		0x7d, 0xc5, 0xc9, 0x35, 0xc9, 0x62, 0x79, 0x93, 0x1d, 0xfc,
		0x55, 0xc6, 0x20, 0x36, 0x95, 0x7c, 0xa1, 0x5d, 0x9c, 0x0f,
		0xdf, 0x95, 0xa7, 0x1f, 0x59, 0xae, 0xb7, 0xa6, 0x39, 0x0f,
		0xd1, 0x7e, 0x93, 0x34, 0x7f, 0xb6, 0x72, 0x96, 0xf2, 0xe3,
		0x20, 0x30, 0x40, 0xe7, 0x77, 0x51, 0xc1, 0x0a, 0x08, 0xf3,
		0xad, 0x86, 0x40, 0xec, 0xf0, 0xd2, 0x9d, 0x88, 0xba, 0xff,
		0x50, 0xf7, 0x4e, 0x88, 0x1f, 0xf0, 0xdd, 0x12, 0x17, 0x05,
		0x97, 0x1e, 0x3c, 0xe3, 0x91, 0x65, 0x10, 0xd9, 0x16, 0x32,
		0x21, 0x44, 0xd8, 0x81, 0xed, 0x17, 0x9a, 0x60, 0x34, 0xee,
		0x4d, 0x81, 0x03, 0xbc, 0x05, 0x56, 0xba, 0x7f, 0xa7, 0x31, /* 100 */
		0xc9, 0x3b, 0x57, 0x5e, 0x38, 0x1c, 0x45, 0xbf, 0xfc, 0x51,
		0x48, 0xf3, 0x05, 0xa9, 0x74, 0x39, 0xdf, 0xa1, 0x34, 0x48,
		0x62, 0x31, 0x5d, 0x58, 0x45, 0x16, 0xc1, 0x9e,	0x26, 0x38,
		0xb4, 0x59, 0x95, 0xdd, 0x92, 0x52, 0x1e, 0x26, 0x20, 0xed,
		0xd2, 0xb2, 0xb3, 0x98, 0x6b, 0xde, 0xbe, 0xf1, 0xa0, 0xbc,
		0x52, 0xc9, 0xfe, 0x97, 0x65, 0x78, 0xd6, 0xce, 0x91, 0xb1,
		0x8e, 0x9b, 0x04, 0xfc, 0x74, 0xb9, 0x1d, 0x52, 0x7d, 0x0b,
		0xf0, 0x1d, 0x2a, 0x3c, 0xde, 0x2e, 0x4c, 0x49, 0xee, 0x62,
		0x9c, 0x59, 0x09, 0x12, 0xd7, 0x4a, 0xd7, 0x0d, 0x03, 0x72,
		0x3d, 0x04, 0x58, 0xd5, 0x19, 0x9e, 0x42, 0x97, 0x2c, 0x32, /* 200 */
		0xb9, 0xda, 0x5c, 0xee, 0xaf, 0x40, 0x44, 0xc9, 0xfd, 0x69,
		0xcf, 0x6e, 0x81, 0x6b, 0x2b, 0xa4, 0x21, 0x3b, 0xde, 0x2d,
		0xc3, 0xb7, 0x03, 0xe3, 0xa3, 0xf2, 0x77, 0xd8, 0x70, 0xf5,
		0x6f, 0x24, 0x48, 0xbf, 0x48, 0xb5, 0xa5, 0xff, 0x86, 0x06,
		0x0c, 0x23, 0xc2, 0xc9, 0x8e, 0xfa, 0x0f, 0xa6, 0xaa, 0x77,
		0x95, 0x1a, 0x1a, 0x91, 0xfe, 0x30, 0x6b, 0x18, 0xc5, 0xfb,
		0x13, 0x76, 0x7a, 0xa5, 0x29, 0xa3, 0x02, 0xce, 0x2b, 0x46,
		0x74, 0x5a, 0x1c, 0xe6, 0x7c, 0x3a, 0xb7, 0x43, 0xea, 0xf9,
		0x3f, 0x53, 0xa9, 0xfb, 0x63, 0x94, 0x26, 0x42, 0x82, 0xdc,
		0x9a, 0x4a, 0xd6, 0xbb, 0xa9, 0xd1, 0x6e, 0xc2, 0xc7, 0x92, /* 300 */
		0x28, 0x3b, 0x33, 0xc3, 0x58, 0xf6, 0xfb, 0x44, 0x5c, 0x5d,
		0xea, 0xe0, 0x62, 0xaa, 0x30, 0x97, 0x30, 0x38, 0x36, 0x47,
		0xb1, 0xae, 0x3b, 0x99, 0x35, 0xab, 0xc9, 0x45, 0x2a, 0x07,
		0xec, 0x34, 0x88, 0x61, 0x70, 0xb0, 0x47, 0xf9, 0xd6, 0x61,
		0x23, 0x48, 0x71, 0x6f, 0x85, 0x3b, 0xf1, 0x8a, 0x17, 0x72,
		0xc1, 0x99, 0x6d, 0x4f, 0x94, 0x2b, 0x11, 0xf2, 0x75, 0x33,
		0x08, 0x67, 0x08, 0xba, 0xa6, 0x50, 0x2a, 0x1f, 0x6b, 0x0e,
		0x38, 0x0e, 0x52, 0xea, 0xe0, 0xd8, 0xcd, 0xd0, 0x11, 0x80,
		0xa7, 0xb9, 0x97, 0xd9, 0x68, 0x1b, 0x21, 0x7c, 0xe2, 0x69,
		0x8e, 0x14, 0x61, 0x39, 0x49, 0x65, 0xbb, 0xc2, 0x7c, 0x2e, /* 400 */
		0x8e, 0xb1, 0x83, 0x28, 0x7b, 0xf0, 0xf0, 0x62, 0xdc, 0x35,
		0x33, 0xea, 0xc9, 0x14, 0x3f, 0x83, 0x60, 0xd3, 0xec, 0x37,
		0xd4, 0xe3, 0x87, 0x86, 0x2c, 0x43, 0x53, 0x80, 0x3f, 0x39,
		0x36, 0x4d, 0x1e, 0x73, 0x2a, 0x5c, 0x8c, 0xfb, 0x2d, 0x2e,
		0x78, 0xd1, 0xf1, 0xe5, 0x8c, 0x89, 0x35, 0x1e, 0x0a, 0xd3,
		0x07, 0x3c, 0x05, 0x0d, 0x4e, 0xa0, 0x81, 0x6a, 0xee, 0xb5,
		0x27, 0xef, 0xa6, 0x2e, 0x04, 0x07, 0x01, 0xf7, 0x40, 0x9b,
		0xbf, 0x7b, 0xf8, 0x3d, 0x0f, 0x68, 0x79, 0xc6, 0x96, 0x16,
		0x53, 0x5b, 0xb0, 0x8f, 0xb1, 0xbd, 0xd4, 0x28, 0xa7, 0x73,
		0x37, 0x7b, 0x06, 0xbf, 0x9c, 0xd2, 0xb4, 0x28, 0x82, 0xb4, /* 500 */
		0xd2, 0x63, 0xf9, 0xb2, 0x56, 0x7c, 0x28, 0x02, 0x4b, 0x91,
		0x63, 0x03, 0xdc, 0x58, 0x6e, 0xb5, 0x88, 0x2e, 0x76, 0x3a,
		0x53, 0x4a, 0xd4, 0x22, 0x7c, 0xa5, 0xca, 0xdc, 0x96, 0x14,
		0xa6, 0xfb, 0x64, 0xe9, 0xa5, 0x09, 0xf2, 0x3e, 0x4d, 0x7e,
		0x40, 0x01, 0x93, 0x05, 0xe0, 0xad, 0x2e, 0x20, 0x7c, 0x4e,
		0x17, 0x45, 0xa8, 0x70, 0x7e, 0xd1, 0xc7, 0xbb, 0x10, 0x94,
		0x81, 0x36, 0x22, 0x62, 0x3f, 0x44, 0xe2, 0xfa, 0x36, 0x9e,
		0x4e, 0x37, 0x75, 0xdf, 0x8d, 0x6f, 0x15, 0x73, 0x41, 0xf0,
		0x6f, 0x32, 0xce, 0xa9, 0xfc, 0x06, 0x8f, 0x39, 0xcf, 0xfb,
		0x34, 0x16, 0xba, 0x74, 0x7d, 0xe6, 0x02, 0x18, 0xd3, 0xe4, /* 600 */
		0x8c, 0xee, 0xd1, 0xef, 0xdc, 0xd4, 0x59, 0x8d, 0xeb, 0x25,
		0x7a, 0x09, 0xc8, 0xcc, 0x38, 0x7b, 0xcb, 0x8b, 0x3c, 0xb8,
		0x77, 0xb4, 0xad, 0x1a, 0x9e, 0x78, 0x20, 0x58, 0x6b, 0x36,
		0x4d, 0x5d, 0xbb, 0x07, 0x38, 0xba, 0x65, 0xf4, 0xfe, 0x4d,
		0x86, 0x94, 0xdf, 0x77, 0x52, 0x08, 0xb1, 0x75, 0x07, 0x85,
		0xaf, 0x77, 0xe9, 0xb3, 0xab, 0x3a, 0x35, 0x30, 0x8a, 0xfd,
		0x34, 0x99, 0x3b, 0x8f, 0xfc, 0x9a, 0x9e, 0xe1, 0x46, 0xcb,
		0xbf, 0xf3, 0xbd, 0xb9, 0x9c, 0x8c, 0x4e, 0x29, 0x6d, 0xa3,
		0x29, 0x5a, 0x1f, 0x88, 0x23, 0xdf, 0x89, 0xa7, 0x88, 0x7c,
		0x90, 0x75, 0x5a, 0xd5, 0x97, 0x60, 0x17, 0xdf, 0x5e, 0x96, /* 700 */
		0x04, 0xf8, 0xae, 0x24, 0xd8, 0xf1, 0xa3, 0x3a, 0xa1, 0x40,
		0x40, 0xd1, 0x45, 0x8e, 0xa5, 0xe5, 0x32, 0x32, 0xcc, 0xcc,
		0x1c, 0x8a, 0x9f, 0x6d, 0xa0, 0xa8, 0x89, 0x9d, 0x75, 0x2b,
		0xd2, 0xc2, 0x07, 0xb2, 0xe1, 0x1a, 0xcd, 0x4f, 0x31, 0x80,
		0xb2, 0x19, 0x5f, 0x06, 0x9d, 0x8e, 0x7a, 0xf1, 0x2e, 0x15,
		0x1d, 0xbc, 0x4c, 0xb4, 0x75, 0x6f, 0x7b, 0xb7, 0xe4, 0x70,
		0xd6, 0x0c, 0xe1, 0x27, 0x93, 0xce, 0x4a, 0x63, 0xde, 0xe7,
		0x48, 0x87, 0x5c, 0x75, 0xae, 0x34, 0x82, 0x5a, 0x71, 0xda,
		0x1f, 0x93, 0x72, 0x64, 0x7c, 0x7b, 0xda, 0x6a, 0xd7, 0xf9,
		0xe8, 0x1f, 0x72, 0x67, 0x71, 0x68, 0x6f, 0x85, 0x64, 0x53, /* 800 */
		0x72, 0x80, 0x8f, 0x1f, 0x7a, 0x15, 0x1c, 0x11, 0xfd, 0x11,
		0xf7, 0x8c, 0xc4, 0x43, 0x41, 0x60, 0x03, 0xf2, 0x79, 0x19,
		0x27, 0xdf, 0x98, 0x37, 0xaf, 0x07, 0xb3, 0x8d, 0x0b, 0xee,
		0x01, 0x85, 0x6f, 0x6d, 0xa9, 0xbd, 0x0e, 0x1f, 0x79, 0xdf,
		0x21, 0x78, 0x24, 0xbf, 0xc5, 0x09, 0x29, 0x9e, 0x64, 0x0d,
		0x8e, 0xff, 0xc9, 0xe7, 0xe8, 0x51, 0xd3, 0xe8, 0xfe, 0xa5,
		0xe0, 0xa9, 0x8d, 0x54, 0x17, 0x3a, 0x34, 0xaf, 0xd9, 0xbd,
		0xfd, 0xaa, 0x8c, 0x81, 0x79, 0xfb, 0x65, 0x85, 0x67, 0x73,
		0x32, 0x2d, 0x32, 0x00, 0xcb, 0x3b, 0xe2, 0x0e, 0x43, 0xe8,
		0x40, 0xf6, 0x42, 0xf2, 0x74, 0x00, 0xa0, 0xfd, 0xe6, 0x96, /* 900 */
		0x4a, 0x2b, 0xbb, 0xa9, 0xd9, 0x95, 0xc4, 0x42, 0x75, 0x12,
		0x3f, 0xbb, 0x79, 0x35, 0x9c, 0x91, 0xb5, 0x24, 0x10, 0xc4,
		0xc5, 0xbd, 0x4a, 0x4c, 0x47, 0xd8, 0x89, 0x92, 0x70, 0xa5,
		0xe6, 0xc9, 0xed, 0x2e, 0xbd, 0x98, 0xc0, 0x17, 0xb0, 0xad,
		0x8c, 0x31, 0x95, 0x81, 0x84, 0x86, 0xb1, 0xaa, 0x42, 0xf6,
		0x2e, 0x10, 0x92, 0x2f, 0x67, 0x73, 0x33, 0xb9, 0x02, 0x43,
		0x52, 0x24, 0x05, 0xdb, 0x9c, 0xec, 0xc5, 0xf1, 0x3e, 0x78,
		0x05, 0xcb, 0x04, 0xd6, 0x91, 0xa8, 0x51, 0x9e, 0x48, 0xa8,
		0xae, 0xa8, 0x8d, 0x13, 0x2d, 0xcd, 0xa1, 0xbe, 0x23, 0x9e,
		0x00, 0x4c, 0x0a, 0x59, 0xf8, 0x18, 0xb0, 0x0a, 0x06, 0xe2, /* 1000 */
		0x0a, 0xb4, 0x16, 0x02, 0xa7, 0x21, 0x4c, 0xac, 0x9a, 0x80,
		0x62, 0x7f, 0xb6, 0xd6, 0xa0, 0x3b, 0x11, 0xd3, 0x30, 0xf9,
		0x3d, 0xfd, 0x26, 0x27)
};
